package tools;

import java.io.*;

/**
 * This is the main class of the HTMLCompressor.
 *
 * The compressor removes all unused characters from
 * an HTML file and makes is shorted.
 *
 * Usage:
 * At this time it works on complete paths. Specify
 * a path at commandline. The compressor create
 * a "compressed" path here and makes a complete compressed
 * copy ofthe site.
 */
public class HtmlCompress
{
    /**
     * The main function.
     *
     * It checks the commandline and calls the compress method.
     *
     * @param args The commandline arguments
     * @see #compress
     */
    public static void main(String[] args)
    {
        System.out.println ("\nHtmlCompress: Html Compressing tool under GPL.");
        System.out.println ("Author: olaf.panz@iname.com");
        if (args.length != 2)
        {
            System.err.println ("Usage: java tools.HtmlCompress <path to compress> <output path>");
        }
        else
        {
            File path = new File (args[0]);
            if (!path.exists ())
            {
                System.err.println ("The specified path does not exist.");
            }
            else
            {
                // create output path
                File outpath = new File (args[1]);
                if (!outpath.exists() && outpath.mkdir() == false)
                {
                    System.err.println ("Cant create path: "+outpath);
                }
                else
                {
                    compress (path,outpath);
                }
            }
        }

        // statistic
        System.out.println ("Read " + _bytesRead +"Bytes.");
        System.out.println ("Wrote " + _bytesWrote +"Bytes.");
        System.out.println ("Compression: " +  (int)(100-((double)_bytesWrote)/((double)_bytesRead)*100) + "%");

        // wait for return, for debug only
        try
        {
            System.in.read();
        }
        catch (Exception ex) {}
    }

    /**
     * This method compress one directory.
     *
     * @param inPath The path to compress
     * @param outPath The output path tp write the result to.
     */
    private static void compress (File inPath,File outPath)
    {
        System.out.println ("Compress: "+ inPath);

        // Iterate over all files here:
        String filelist[] = inPath.list();
        File f;
        int i;

        for (i=0;i<filelist.length;i++)
        {
            // create file object to examin the type:
            f = new File (inPath,filelist[i]);
            File output = new File (outPath,filelist[i]);
            if (f.isDirectory())
            {
                // create output subdir:
                if (!output.exists() && output.mkdir () == false)
                {
                    System.err.println ("Can't create: "+output);
                }
                else
                {
                    compress (f,output);
                }
            }
            else
            {
                // test, if it is a HTML file:
                int iPos = filelist[i].lastIndexOf ('.');

                if (iPos != -1)
                {
                    String ext = filelist[i].substring (iPos+1).toLowerCase();

                    if ("html".equals (ext) || "htm".equals (ext) || "xml".equals (ext))
                    {
                        compressFile (f,output);
                    }
                    else
                        copyFile (f,output);
                }

            }
        }
    }

    /**
     * Compress a HTML file.
     *
     * @param in The input file
     * @param out The output file
     */
    private static void compressFile (File fin,File fout)
    {
        InputStream in = null;
        HtmlCompressorOutputStream out = null;
        System.out.println ("Compress file:"+fin);

        try
        {
            in = new FileInputStream (fin);
            out = new HtmlCompressorOutputStream (new FileOutputStream (fout));

            long len = fin.length ();
            _bytesRead += len;

            for (;len > 0;len--)
                out.write (in.read());

            _bytesWrote += out.getWrittenBytes ();

        }
        catch (IOException ex)
        {
            System.err.println (ex.toString ());
        }
        finally
        {
            if (in != null)
            {
                try
                {
                    in.close ();
                }
                catch (IOException ex)
                {
                    System.err.println (ex.toString ());
                }
            }
            if (out != null)
            {
                try
                {
                    out.close ();
                }
                catch (IOException ex)
                {
                    System.err.println (ex.toString ());
                }
            }
        }

    }

    /**
     * Copy a file.
     *
     * @param in The input file
     * @param out The output file
     */
    private static void copyFile (File in,File out)
    {
        System.out.println ("Copy file:"+in);

        FileOutputStream fo = null;
        FileInputStream fi = null;

        try
        {
            fi = new FileInputStream (in);
            fo = new FileOutputStream (out);

            long len = in.length();
            byte []buf = new byte[(int)len];
            _bytesRead += len;
            _bytesWrote += len;

            fi.read (buf);
            fo.write (buf);
        }
        catch (IOException ex)
        {
            System.err.println (ex.toString());
        }
        finally
        {
            if (fo != null)
            {
                try
                {
                    fo.close ();
                }
                catch (IOException ex)
                {
                    System.err.println (ex.toString());
                }
            }
            if (fi != null)
            {
                try
                {
                    fi.close ();
                }
                catch (IOException ex)
                {
                    System.err.println (ex.toString());
                }
            }
        }

    }

    private static long _bytesRead = 0;
    private static long _bytesWrote = 0;


}
