package de.jbible.core;

import java.io.*;
import java.util.*;

import de.jbible.bible.*;

/**
 * This describes the interface
 * of a bible text. BibleTests are
 * provided by a
 * {@link BibleTextProviderService BibleTextProviderService}
 */

public interface BibleText
{
    /**
     * Index to the title of this translation.
     *
     * This is an index to the sentence list.
     *
     * @return The title index.
     */
    public String getTitle ();

    /**
     * Index to the copyright notice of this translation.
     *
     * This is an index to the sentence list.
     *
     * @return The copyright notice index.
     */
    public String getCopyright ();

    /**
     * Get the Copyright identifier.
     *
     * This identifier specify what is allowd to do with this text.
	 *  <b>Bit 0-3:</b><br>
	 *<ul>
	 *<li>  1: The bible translation is public domain.</li>
	 *<li>  2: The bible translation is shareware.</li>
	 *<li>  3: The bible translation is normal buy ware</li>
	 *</ul>
	 *<ul>
	 *<b>Other bits:</b><br>
	 *<li>  Bit 4 == 1: It is allowed to publish the translation over the web
	 *          (As ASCII text in an web application, not as one file.)</li>
	 *<li>  Bit 5 == 1: It is allowed to copy the translation to friends</li>
	 *</ul>
     *
     * @return The copyright identifier
     */
	public int getCopyrightIdentifier ();


    /**
     * Return a Java conform language identifier
     * that describe the language of this bible translation.
     *
     * See Java class Local for details.
     *
     * @return The identifer.
     */
	public String getLocale ();


    /**
     * Returns the book title
     *
     * @param bookId The id of the book as defined in BibleBooks.
     * @return The title of the book as string.
     */
	public String getBookTitle (byte bookId);

    /**
     * Access to all books.
     *
     * This method returns a list of all books of this
     * translation. The array contains book identifier as
     * defined in BibleBooks.
     *
     * @return The book identifier array.
     * @see de.jbible.bible.BibleBooks
     */
	public byte [] getBooks ();

    /**
     * Get the chapter count of a book.
     *
     * @param book The book identifier as defined in BibleBooks
     * @return The count of chapters of this book.
     * @see BibleBooks
     */
	public int getChapterCount (byte book);

    /**
     * Get the vers count of a chapter.
     *
     * @param book The book identifier as defined in BibleBooks
     * @param chapter The chapter of the book the vers count is requested for.
     * @return The count of chapters of this book.
     * @see BibleBooks
     */
	public int getVersCount (byte book,int chapter);


    /**
     * Access a vers.
     *
     * You get the vers as String array. To create the whole
     * sentence, use the TextCompositor class.
     *
     * @see TextCompositor
     * @param book The book identifier as defined in BibleBooks
     * @param chapter The chapter of the book the vers count is requested for.
     *		It is relative to the book.
     * @param vers The vers number, relative to the chapter.
     * @return The vers array.
     * @exception IndexOutOfBounds The location does not exists in
     *    this bible.
     */
	public List getVersStrings (byte book,int chapter,int vers);


    /** Copyright define: Text is public domain. */
    public static final int COPYRIGHT_PUBLICDOMAIN = 1;

    /** Copyright define: Text is shareware. */
    public static final int COPYRIGHT_SHAREWARE = 2;

    /** Copyright define: Text is buyware. */
    public static final int COPYRIGHT_BUYWARE = 3;

    /** Copyright define: Text can published over the web. */
    public static final int COPYRIGHT_WEBPUBLISH = 16;
    /** Copyright define: Text can give to friends. */
    public static final int COPYRIGHT_TOFRIENDS = 32;

}
