package de.jbible.core;

import java.io.*;
import de.jbible.bible.*;


/**
 * This interface must be implemented by ImportFilter for bible translations.
 *
 * A Import Dialog Service provides all GUI stuff for
 * this service. It lets the user select a import filter, shows the
 * description and let him choose a file. Then it copies all files
 * that belongs to this import to a temporary directory. So
 * the filter has all together.
 *
 * @see #getFileFilter
 * @see #getFileList
 */

public interface ImportFilterService extends Service
{
    /**
     * Returns the title(s) of the bible translation(s) this filter works with.
     *
     * A filter can handle more then one translation.  These strings are keys
     * for translation.
     *
     * @return An array of titles.
     */
    public String[] getTranslationTitle ();

    /**
     * Returns the description of this filter.
     *
     * The text must describe how to use the filter. These strings are keys
     * for translation.
     *
     * @return An array of descriptions.
     */
    public String[] getDesription ();

    /**
     * Datatype that is imported by this filter.
     *
     * At this time there is only one data type: Bible Translation.
     * But there will be more types in the future, e.B. HTML Book
     *
     * @return An array with all datatypes of
     * @see #BIBLETRANSLATION
     */
    public int [] getDataTypes ();

    /**
     * Test, if the given Path/File can imported from this filter.
     *
     * @param f The file that should be tested.
     * @param index The index of the selected import method
     * @return true, if the file is or might be Ok, else false
     */
    //public boolean doTest (File f,int index);

    /**
     * Creates a BibleReaderObject that imports a bible translation
     * from given path/file.
     *
     * @param f The file that should be imported.
     * @param index The index of the selected import method
     * @exception IOException An io error might occure.
     * @return The Reader object that reads the translation.
     */
    public BibleImportReader createImportReader (File f,int index)
        throws IOException;

    /**
     * Returns a file filter for the search dialog.
     *
     * You must create a FileFilter, if you want to select a file.
     * If this method returns null, a path dialog is shown.
     *
     * @param index The index of the selected import method
     * @return The FileFilter object
     */
    public FileFilter getFileFilter (int index);

    /**
     * Access to a list of all files that belong to this translations.
     *
     * The list contains a row for each file and 3 columns:
     *
     * Col 1: The filename
     * Col 2: The file size null if it should not be tested.
     * Col 3: null

     * @param index The index of the selected import method
     * @return The list of all files for this translation
     */
    public Object[][] getFileList (int index);


    /** Datatype bible translation. */
    public static final int BIBLETRANSLATION = 1;
}
