package de.jbible.core;

import java.util.*;
import java.io.*;

/**
 * The logging manager proceeds errors and logging messages.
 *
 * All error messages from the whole program should be proceed here to
 * use the correct services for further proceeding of the messages.
 * The LoggingManager sends the messages to all services that
 * implements the {@link de.jbible.core.DebugService DebugService} interface.
 *
 * @see de.jbible.core.DebugService
 */

public class LoggingManager
{
    /**
     * This method logs an error.
     *
     * The most exception from the java runtime should be
     * proceeded here.
     *
     * @param msg The error message, should describe the error context
     * @param ex The exception that was thrown
     * @param module The module class. If
     * you need a new or more specific module name, feel free to add it there.
     */
    public static void error (String msg,Throwable ex,Object module)
    {
        StringWriter sw = new StringWriter ();
        PrintWriter pw = new PrintWriter(sw);
        ex.printStackTrace(pw);
        sw.flush();

        proceed (new LogMessage (msg+ex+"\n"+sw.getBuffer(),module,MSG_ERROR));
    }

    /**
     * This method logs an error.
     *
     * Use this method if you have no exception.
     *
     * @param msg The error message, should describe the error context
     * @param module The module class. If
     * you need a new or more specific module name, feel free to add it there.
     */
    public static void error (String msg,Object module)
    {
        proceed (new LogMessage (msg,module,MSG_ERROR));
    }


    /**
     * This method logs a logging message.
     *
     * Use logging messages if you want to see how your module
     * works.
     *
     * @param msg The error message, should describe the error context
     * @param module The module class.  If
     * you need a new or more specific module name, feel free to add it there.
     */
    public static void log (String msg,Object module)
    {
        proceed (new LogMessage (msg,module,MSG_LOG));
    }

    /**
     * Initialisation of the logging manager.
     *
     * This method is called from the TheServiceManager if all
     * services are registed.
     */
    static void init (TheServiceManager man)
    {
        _debugServices = man.getServices (de.jbible.core.DebugService.class);

        // proceed all Messaged that are created until now:
        ListIterator iter = _bufferLst.listIterator();

        while (iter.hasNext())
        {
            // proceed again:
            proceed ((LogMessage)iter.next());
        }

        // loose buffer:
        _bufferLst = null;

    }
    /**
     * Internal proceeding of a message.
     *
     * @param module The module name. This must be constant from
     * this class (see below). If
     * you need a new or more specific module name, feel free to add it there.
     */
    private static void proceed (LogMessage msg)
    {

        if (_debugServices == null)
        {
            if (_bufferLst != null)
            {
                // we are not initialised, add messages to List:
                _bufferLst.add(msg);
            }
            else
            {
                // bufferlist is after the initializing null, if no DebugService
                // is registered...
            }
        }
        else
        {
            // proceed to services:
            ListIterator iter = _debugServices.listIterator();
            DebugService db;

            while (iter.hasNext())
            {
                db = (DebugService)iter.next();
                db.message (msg);
            }
        }
    }

    /**
     * Buffer for debug messages that comes before the object
     * is initialized.
     */
    private static List _bufferLst = new LinkedList ();

    /**
     * List of all Services that implements the
     * DebugService interface.
     */
    private static List _debugServices = null;

    /** Message type error. */
    public static final int MSG_ERROR = -1;

    /** Message type log. */
    public static final int MSG_LOG = -2;
}
