package de.jbible.core;

import java.util.*;

/**
 * This is the connection between the ServiceManager and the Service.
 *
 * Each Service has its own ServiveContext object. It handles all its
 * actions over its context.
 *
 * @see ServiceManager
 * @see Service
 */
public class ServiceContext
{

    /**
     * Constructor.
     *
     * @param name The complete class name of the service.
     * @param man The service manager
     */
    ServiceContext(Service service,TheServiceManager man)
    {
        _service = service;
        _man = man;
        _name = service.getClass().getName();
    }

    /**
     * Retrieve a property for the context.
     *
     * Note: All properties are pending to the context, so
     * the key "A" from service b returs a different result than
     * key "A" from service c.
     *
     * @param key The key to retrive the property for.
     * @return The value for the key. This might be null, if
     * no property is set.
     */
    public String getProperty (String key)
    {
        return _man.getProperties().getProperty (_name+key);
    }

    /**
     * Set a property.
     *
     * Note: All properties are pending to the context, so
     * the key "A" from service b returs a different result than
     * key "A" from service c.
     *
     * @param key The key to retrive the property for.
     * @param value The value string to set.
     */
    public void setProperty (String key,String value)
    {
        _man.getProperties().put (_name+key,(value!=null)?value:"");
    }


     /**
      * Access the list of services that imeplements a special class.
      *
      * It returns an list with all services that implements the
      * class specified in the parameter cls.
      *
      * @param className The class that must be implemented.
      * @return A vector with the found services, could be null.
      */
     public List getServices (Class className)
     {
        return _man.getServices (className);
     }


    /**
     * Exit this program.
     *
     * Method is called from the GUI to terminate the program.
     *
     * @param exitCode The termination code
     */
    public void exit (int exitCode)
    {
        _man.exit (exitCode);
    }

    /**
     * Access to the actual locale object.
     *
     * The locale object is stored in the ServiceManager
     *
     * @return The actual locale object.
     * @see ServiceManager
     */
    public Locale getLocale ()
    {
        return _man.getLocale();
    }

    /**
     * Change the actual locale.
     *
     * The locale object is stored in the ServiceManager
     *
     * @see ServiceManager
     * @param locale The new locale to set.
     */
    public void setLocale (Locale locale)
    {
        _man.setLocale (locale);
    }


    /**
     * Translate a string into the current locale language.
     *
     * The method requests the translation from the
     * ServiceManager.
     *
     * @see ServiceManager
     * @param key The value to translate
     * @return The tanslated string.
     */
    public Object translate (String key)
    {
        return _man.translate (key);
    }

     /**
      * Returns an array of supported locales.
      *
      * We are supporting just a few locales. The list
      * of supported locales is stored in the property file.
      *
      * @return The array of supported locales.
      */
    public Locale[] getSupportedLocales ()
    {
        return _man.getSupportedLocales ();
    }

    /**
     * Set the name of the service.
     *
     * This method should be called from the init method of a service.
     *
     * @param title The name of the service.
     */
    public void registerTitle (String title)
    {
        _title = title;
    }

    /**
     * Set the copyright of the service.
     *
     * This method should be called from the init method of a service.
     *
     * @param copyright The copyright of the service.
     */
    public void registerCopyright (String copyright)
    {
        _copyright = copyright;
    }

    /**
     * Set the resource bundle of the service.
     *
     * This method should be called from the init method of a service.
     *
     * @param resourceBundle The resource bundle of the service.
     */
    public void registerResourceBundle (String resourceBundle)
    {
        _resourceBundle = resourceBundle;
    }

    /** The service that belongs to this context.   */
    Service _service;

    /** The complete name of the service.
     *
     * This is the class name.
     */
    String _name;

    /** The title of the service. */
    private String _title;


    /** The copyright of the service.   */
    String _copyright = DEFAULT_COPYRIGHT;

    /** The copyright of the service.   */
    public static final String DEFAULT_COPYRIGHT =
        "This module is under the GNU public license (GPL). "+
        "See www.jbible.de for details.";

    /** The optional resource bundle. */
    String _resourceBundle;


    /** Reference to the service manager.  */
    private TheServiceManager _man;
}
