package de.jbible.core;

import java.util.*;
import java.io.*;

/**
 * The module manager manages all modules of the bible.
 *                                                           3
 * Invoke the {@link ServiceManager#start start} method to start
 * the service manager.
 *
 * A service is:
 * <ol>
 * <li><b>A bible text provider:</b> This service must also implement the
 * BibleTextProviderService
 * interface. A bible text provider is a service which provides several
 * bible translations.</li>
 *
 * <li><b>A bible text viewer:</b>  This service must also implement the
 * BibleTextViewerService
 * interface. A bible textviewer is able to view a bible translation
 * provided by a bible text provider.</li>
 *
 * <li><b>GUI aware service:</b> This service must also implement the
 * {@link WindowService WindowService} interface.
 * A GUI aware service is a service that can be called by the module.
 * For example the dialog to find a vers is a GUI aware service.</li>
 *
 * <li><b>HTML Text provider:</b> This service must also implement the
 * HTMLTextService
 * It provides books and missionary texts around the bible in HTML</li>
 *
 * <li><b>Bible links provider:</b> This service must also implement the
 * BibleLinkProviderService
 * interface. It provides several link system between the bible verses. For
 * example the Luther'84 translation cames with links. So the provider of the
 * text can also provide the links so that you can use them with other
 * translations.</li>
 *
 * <li><b>User defined services:</b> In addition you can create your own
 * services with your own specialized interfaces.</li>
 * </ol><br>
 * Services are distinguished by the interfaces they are implementing.
 *
 * <br><b>Locale management</br>
 *
 * The ServiceManager knows the actual used locale. The locale
 * describe the localisation where the program actually runs.
 *
 * <H2>Nameing convention for services </H2>
 * A service class must be located in a subpackage of the package
 * {@link de.jbible.service de.jbible.service}
 * The package must be named as the Service class in small letters,
 * e.G is the service
 * has the name MainWindow, the package must be named mainwindow.
 *
 * <H2>Property file</H2>
 * The entry "services=" specifies the services that will be load.
 * enter only class names. The rest will be generated.
 */

class TheServiceManager
{
    /**
     * Constructor.
     *
     * Initialize the Service manager, load and start the services.
     * The service manager is a singleton. Its created from the main
     * method.
     *
     * @param args The command line parameters
     */
     public TheServiceManager (String[] args)
     {
     	initStaticProperties (args);

        // read property file from the current directory:
        String propfile = System.getProperty ("user.dir") +
                         File.separatorChar +
                         SM_PROPFILE;

        InputStream in = null;

        try
        {
            try
            {
                in = new FileInputStream (propfile);
            }
            catch (Exception x)
            {
                // next: try to load it from the resource:
                in = ClassLoader.getSystemResourceAsStream ("de/jbible/default.prop");
            }

            if (in != null)
                _props.load (in);
        }
        catch (IOException ex)
        {
            LoggingManager.error ("Try to load property file: "+ propfile,ex,
                this);
        }
        finally
        {
            if (in != null)
            {
                try
                {
                    in.close ();
                }
                catch (IOException x) {}
                finally
                {
                    in = null;
                }
            }
        }

        // find the correct locale for the environment
        // Algorithm:
        // 1.) Look if a locale is stored in the properties, if yes,
        //      use it.
        // 2.) If that doesn't fit, get system default and search for
        //      a supported locale for it.
        // 3.) If that also doesn't fit, use english.

        _locale = mapLocale (_props.getProperty (SM_LOCALE));
        if (_locale == null)
        {
            _locale = mapLocale (Locale.getDefault().getLanguage());
            if (_locale == null)
                _locale = Locale.ENGLISH;
        }


        // create servicelist from propfile
        LinkedList services = new LinkedList ();

        String serviceClasses = _props.getProperty ("services");

/*        if (serviceClasses == null || "".equals(serviceClasses))
        {
        }*/

	        String name = null;
        int pos =-1;
        int oldPos = -1;
        Class cls = null;
        Class serviceCls = null;
        Service sve = null;

        try
        {
            serviceCls = Class.forName ("de.jbible.core.Service");
        }
        catch (ClassNotFoundException ex)
        {
            // This error cant occure because Service is referenced from
            // this module (thanks to Joe Walker)
            /*LoggingManager.error ("Try to instanciate interface:"+
                "de.jbible.core.Service",ex,
                LoggingManager.LOC_MAIN);
            System.err.println ("Fatal error: some system classes are "+
                "missing!");

            System.exit(1); // don't go further!*/
        }

        String bundle;
        // instanciate registered services

        if (serviceClasses != null)
        {
            do
            {
                oldPos = pos+1;
                pos = serviceClasses.indexOf (',',oldPos);

                if (pos == -1)
                    name = serviceClasses.substring (oldPos);
                else
                    name = serviceClasses.substring (oldPos,pos);

                // load the service class:

                try
                {
                    cls = Class.forName ("de.jbible.service."
                        +name +"."+name);
                }
                catch (ClassNotFoundException ex)
                {
                    LoggingManager.error ("Try to instanciate service:"+name,ex,
                        this);
                    continue; // dont go further!
                }

                // check, if service object implents the Service interface
                if (!serviceCls.isAssignableFrom (cls))
                {
                    LoggingManager.error ("The service class "+name+
                        " does not implement the Service interface!",
                         this);
                    continue;

                }

                try
                {
                    // create service instance
                    sve = (Service) cls.newInstance ();
                }
                catch (Exception ex)
                {
                    LoggingManager.error ("Cant instanciate the service "+name,ex,
                        this);
                }

                services.add(sve);

                // get resource bundle:
/*                bundle = sve.getResourceBundle ();
                if (bundle != null)
                {
                    // store it in the resource bundle list
                    _resourceBundleLst.add (bundle);
                }*/

            }
            while (pos != -1);
        }


        // now we must call the init methods from all services:
        ListIterator iter = services.listIterator();
        ServiceContext context;

        while (iter.hasNext())
        {
            sve = (Service)iter.next();

            // create Service context for service:
            context = new ServiceContext (sve,this);
            _contextList.add(context);

            sve.init (context);
        }

        //merge all ResourceBundleLists from all of the actual resources
        mergeResourcBundleLists ();

        // Initialise the logging manager: To handle Debug services we must
        //  init the LoggingManager so that it can work correct.
        LoggingManager.init (this);


        // Start the application now:
        iter = services.listIterator();

        while (iter.hasNext())
        {
            sve = (Service)iter.next();
            sve.start ();

            LoggingManager.log("Start service: "+sve.getClass().getName(),
                this);
        }


        LoggingManager.log("Init complete!",
                this);

     }

    /**
     * Map a given locale string to the supported locales.
     *
     * @param language The language to find
     * @return The mapped loacale object or null, if no locale mapped
     */
    private Locale mapLocale (String language)
    {
        Locale loc[] = getSupportedLocales ();

        for (int i=0;i<loc.length;i++)
        {
            if (loc[i].getLanguage().equals (language))
                return loc[i];
        }
        return null;
    }

    /**
     * Access to the properties object.
     *
     * @return The global properties object.
     */
    Properties getProperties ()
    {
        return _props;
    }

    /**
     * Access to the selected locale.
     *
     * @return The actual selected locale.
     */
    Locale getLocale ()
    {
        return _locale;
    }

    /**
     * Set the actual locale.
     *
     * This method will inform all services that have implement
     * the {@link LocaleChangeListener LocaleChangeListener} interface
     * so that they can change their display.
     *
     * @param locale The new locale object.
     */
     void setLocale (Locale locale)
     {
        _locale = locale;
        mergeResourcBundleLists ();

        // inform all listeners
        List l = getServices (de.jbible.core.LocaleChangeListener.class);

        if (l != null)
        {
            ListIterator iter = l.listIterator();
            LocaleChangeListener listener;

            while (iter.hasNext())
            {
                listener = (LocaleChangeListener)iter.next();
                listener.localeChanged (_locale);
            }
        }

        // store new locale in property file:
        _props.setProperty (SM_LOCALE,_locale.toString());
     }

     /**
      * Returns an array of supported locales.
      *
      * We are supporting just a few locales. The list
      * of supported locales is stored in the property file.
      *
      * @return The array of supported locales.
      */
    Locale[] getSupportedLocales ()
    {
        return _supportedLocales;
    }

     /**
      * Translate a object.
      *
      * The method uses all registerd resource bundles to get
      * a translation for the string.
      *
      * @param key The string to tranlate.
      * @return The translation
      * @see ServiceContext#translate
      */
     Object translate (String key)
     {
        Object str = _mergedResourceBundle.get (key);
        if (str != null)
            return str;

        return key;
     }

     /**
      * Merge all registered resource bundles to one.
      */
    private void mergeResourcBundleLists ()
    {
        // clear old map:
        _mergedResourceBundle.clear ();

        Iterator iter = _contextList.iterator();
        String name,key;
        Object value;
        ResourceBundle bundle = null;
        Enumeration enum;

        while (iter.hasNext())
        {
            name = ((ServiceContext)iter.next())._resourceBundle;

            // did this service has a resource bundle?
            if (name != null)
            {
                try
                {
                    bundle = ResourceBundle.getBundle (name,_locale);
                }
                catch (Exception x)
                {
                    LoggingManager.error ("Cant load Resource Bundle",x,
                        this);
                }

                if (bundle != null)
                {
                    enum = bundle.getKeys ();

                    while (enum.hasMoreElements())
                    {
                        key = (String)enum.nextElement ();
                        value = bundle.getObject (key);
                        _mergedResourceBundle.put (key,value);
                    }
                }
            }
        }
    }

     /**
      * Access the list of services that implements a special class.
      *
      * It returns a list with all services that implements the
      * class specified in the parameter cls.
      *
      * @param cls The class that must be implemented.
      * @return A list containing the found services, could be null.
      */
     List getServices (Class cls)
     {
        List resultLst = null;

        ListIterator iter = _contextList.listIterator();

        while (iter.hasNext())
        {
            ServiceContext obj = (ServiceContext)iter.next();
            if (cls.isAssignableFrom (obj._service.getClass()))
            {
                if (resultLst == null)
                    resultLst = new LinkedList ();

                // add to list:
                resultLst.add (obj._service);
            }
        }

        return resultLst;

     }

     /**
      * Access the list of services that imeplements a special class.
      *
      * It returns an list with all services that implements the
      * class specified in the parameter cls.
      *
      * @param className The class that must be implemented.
      * @return A vector with the found services, could be null.
      */
     /*List getServices (String className)
     {
        try
        {
            Class cls = Class.forName (className);
            return getServices (cls);
        }
        catch (ClassNotFoundException ex)
        {
            LoggingManager.error ("Try to get class object for:"+className,
                ex,this);
        }
        return null;
     }*/

    /**
     * Terminate the program.
     *
     * The method is called from the context
     */
    void exit (int exitCode)
    {
        // call exit for each service
        // now we must call the init methods from all services:
        ListIterator iter = _contextList.listIterator();

        while (iter.hasNext())
        {
            ((ServiceContext)iter.next())._service.exit();
        }

        // write back properties
         // read property file from the current directory:
        String propfile = System.getProperty ("user.dir") +
                         File.separatorChar +
                         SM_PROPFILE;

        FileOutputStream out = null;

        try
        {
            out = new FileOutputStream (propfile);
            _props.store(out,"JBible");
        }
        catch (IOException ex)
        {
            LoggingManager.error ("Try to load property file: "+ propfile,ex,
                this);
        }
        finally
        {
            if (out != null)
            {
                try
                {
                    out.close ();
                }
                catch (IOException x) {}
                finally
                {
                    out = null;
                }
            }
        }

        // exit!
        System.exit (exitCode);
    }

    /**
     * Initalize static properties.
     *
     * The static props may later be initalized by
     * command line arguments.
     *
     * @param args The commandline arguments
     */

	private void initStaticProperties (String [] args)
    {
    	// Create directory for bible translations:
		String userDir = System.getProperty ("user.dir");
        String bibDir = userDir + File.separatorChar +
        	"bibleTranslations";
        File f = new File (bibDir);
        if (!f.exists())
        	f.mkdirs();
		ServiceManager.setStaticProperty("BIBLEPATH",f.getAbsolutePath());

    }



    /** Name of the property filee.*/
    private final static String SM_PROPFILE = ".bible.prop";
    /** Constant to store the actual locale: */
    private final static String SM_LOCALE = "Locale";

    /** List of all service context. */
    private List _contextList = new LinkedList ();

    /** The properties of the application.  */
    private Properties _props = new Properties ();

    /** The actual locale of the application.  */
    private Locale _locale = Locale.getDefault();

    /** Merged resource bundle. */
    private Map _mergedResourceBundle = new HashMap ();
    /** The list of all supported locales. */
    private static final Locale[] _supportedLocales = {
        Locale.ENGLISH,Locale.GERMAN };

}

