package de.jbible.service.ImportDBG;

import java.io.*;
import java.util.*;
import de.jbible.bible.*;
import de.jbible.core.*;
import de.jbible.tool.misc.*;
import de.jbible.tool.bibleimport.*;



/**
 * Import Reader to import the Luther 84 ASCII text.
 */

public class ImportLuther84 implements BibleImportReader
{
	/** Not allowed. */
	private ImportLuther84 () {}

    /**
     * Create a new reader.
     *
     * @param path The path where to find all files.
     */
	public ImportLuther84 (File path)
    {
    	_path = path.getParent();
    }

    /**
     * Read the next book.
     *
     * <p>
     * If the value is null, you are behind the last book of
     * the stream.</p>
     *
     * @return The complete book as object tree.
	 * @exception IOException An io exception occures while reading.
     */
    public Object readBook () throws IOException
	{
        if (open (++_index) == true)
		{

            ImportDBGParser parser = new ImportDBGParser();

            try
            {
                Book b = (Book)parser.yyparse(_lexer);
                b.setBookId(((Byte)_books[_index][0]).byteValue());

                // The first headline is the title of this book:
                Iterator iter = b.getMetaInfo();
              	MetaInfo mf;

                if (iter == null)
                {
                	// fix for bad file format:
                    iter = b.getChapters();
                    Chapter c = (Chapter)iter.next();
                    iter = c.getMetaInfo();
                    if (iter == null)
                    {
                    	iter = c.getVerses();
                        de.jbible.tool.bibleimport.Vers v =
                        		(de.jbible.tool.bibleimport.Vers)iter.next();
                        iter = v.getMetaInfo();
                    }
                }

                while (iter.hasNext())
                {
                	mf = (MetaInfo)iter.next();
                    if (mf.getType() == MetaInfo.HEADLINE)
                    {
                    	b.setName (mf.getSentence());
                        break;
                    }
                }

                //System.out.println (b.toString());

                return b;

            }
            catch (Throwable t)
            {
                LoggingManager.error ("Error while parsing Luther'84",
                    t,this);
                LoggingManager.log(Debug.report(),this);
                throw new IOException (t.toString());
            }
 		}
        
        return null;
	}

    /**
     * Returns the filename of the file after import.
     *
     * This filename has no extension. That means that
     * an extension will be added.
     *
     * @return The file name.
     */
	public String getFileName ()
    {
    	return "Luther84";
    }

    /**
     * Get the copyright notice.
     *
     * @return The copyright string.
     */
	public String getCopyright ()
    {
    	return
         "Wir weisen Sie ausdrcklich darauf hin, da ein Abdruck des "+
		 "Textes und Vervielfltigung auf elektronischen Datentrgern "+
		 "oder audiovisuellen Medien jeder Art zum gewerblichen Gebrauch "+
   		 "nur mit Genehmigung des Rates der Evangelischen Kirche in "+
		 "Deutschland (EKD) erlaubt ist.\n\n"+
         "(c) 1989 Deutsche Bibelgesellschaft, Stuttgart\n\n"+
		 "Wir bitten Sie auerdem, auch von Kopien fr Ihre Bekannte und "+
   		 "Verwandte Abstand zu nehmen und wnschen Ihnen viel Freude am Bibeltext.\n"+
		 "Deutsche Bibelgesellschaft  9/89";
    }

    /**
     * Get the copyright identifer as defined in
     * {@link de.jbible.tool.bibleimport.Header Header}
     *
     * @return The copyright identifier.
     */
	public int getCopyrightIdentifier ()
    {
    	return BibleText.COPYRIGHT_BUYWARE;
    }

    /**
     * Get the original title of the bible translation.
     *
     * @return The original title.
     */
    public String getTitle ()
    {
    	return "Lutherbibel '84 mit Apokryphen";
    }


    /**
     * Open the file with the specified index with the lexer.
     *
     * @param index The index of the file to open.
     * @return false, if we are at the end of the bible.
     * @exception IOException An io exception occures while reading.
     */
    private boolean open (int index) throws IOException
    {
    	if (index >= _books.length)
        	return false;

        File file = new File (_path,((String)_books[index][1])+_ext);
        Debug._fileName = file.toString();

    	_lexer = new _Yylex (new InputStreamReader (
                new FileInputStream (file),
                "Cp850"),this);
        return true;

    }

	public void close() throws IOException
	{
		//TODO: implement this de.jbible.bible.BibleImportReader method;
	}

    /** Path where I can find the source files.
     *
     * Each Book is one file.
     */
    private String _path;

    /** Index of actual read file. */
    private int _index = -1;

    /** Actual tokenizer. */
    private _Yylex _lexer = null;


    /**
     * Create the import file list for Luther'84.
     *
     * The list contains a row for each file and 3 columns:
     *
     * Col 1: The filename
     * Col 2: The file size null if it should not be tested.
     * Col 3: null
     *
     * @return The file list.
     */
    static Object [][] createFileList ()
    {
        Object [][] result = new Object [_books.length][3];
        int i;
        for (i=0;i<result.length;i++)
        {
            result[i][0] = _books[i][1]+_ext;
            result[i][1] = _books[i][2];
        }

        return result;

    }

    /**
     * Return a Java conform language identifier
     * that describe the language of this bible translation.
     *
     * See Java class Local for details.
     *
     * @return The identifer.
     */
	public String getLocale ()
    {
    	return Locale.GERMAN.toString();
    }


    /** Extension of source files. */
    private static final String _ext = ".TXT";

    /**
     * Create a file filter to select the first file from this
     * translation.
     *
     * @return The file filter.
     */
    static FileFilter createFileFilter ()
    {
        return new SingleFileFilter (((String)_books[0][1])+_ext,((Long)_books[0][2]).longValue());
    }

	/**
     * Number of last read token.
     *
     * This is used to fix the mistakes in the text.
     */
    public int _tokenNr=0;

    /**
     * Array of all books with corresponding file name and file size.
     */
    private static final Object [] []_books = {
        {new Byte (BibleBooks.Genesis),"1_MOSE",new Long(220667)},
        {new Byte (BibleBooks.Exodus),"2_MOSE",new Long(184194) },
        {new Byte (BibleBooks.Leviticus),"3_MOSE",new Long(134369)},
        {new Byte (BibleBooks.Numbers),"4_MOSE",new Long(180141)},
        {new Byte (BibleBooks.Deuteronomy),"5_MOSE",new Long(168353)},
        {new Byte (BibleBooks.Joshua),"JOSUA",new Long(106705)},
        {new Byte (BibleBooks.Judges),"RICHTER",new Long(107339)},
        {new Byte (BibleBooks.Ruth),"RUT",new Long(14730)},
        {new Byte (BibleBooks.Samuel1),"1_SAMUEL",new Long(142934)},
        {new Byte (BibleBooks.Samuel2),"2_SAMUEL",new Long(118083)},
        {new Byte (BibleBooks.Kings1),"1_KOENIG",new Long(136220)},
        {new Byte (BibleBooks.Kings2),"2_KOENIG",new Long(132892)},
        {new Byte (BibleBooks.Chronicles1),"1_CHRONI",new Long(121035)},
        {new Byte (BibleBooks.Chronicles2),"2_CHRONI",new Long(149435)},
        {new Byte (BibleBooks.Ezra),"ESRA",new Long(41823)},
        {new Byte (BibleBooks.Nehemiah),"NEHEMIA",new Long(60839)},
        {new Byte (BibleBooks.Esther),"ESTER",new Long(32081)},
        {new Byte (BibleBooks.Job),"HIOB",new Long(112208)},
        {new Byte (BibleBooks.Psalms),"PSALM",new Long(266609)},
        {new Byte (BibleBooks.Proverbs),"SPRUECHE",new Long(95461)},
        {new Byte (BibleBooks.Ecclesiastes),"PREDIGER",new Long(32600) },
        {new Byte (BibleBooks.SongOfSongs),"HOHELIED",new Long(15093) },
        {new Byte (BibleBooks.Isaiah),"JESAJA",new Long(222743) },
        {new Byte (BibleBooks.Jeremiah),"JEREMIA",new Long(251352) },
        {new Byte (BibleBooks.Lamentations),"KLAGEL",new Long(20686) },
        {new Byte (BibleBooks.Ezekiel),"HESEKIEL",new Long(224392) },
        {new Byte (BibleBooks.Daniel),"DANIEL",new Long(67793) },
        {new Byte (BibleBooks.Hosea),"HOSEA",new Long(32881) },
        {new Byte (BibleBooks.Joel),"JOEL",new Long(12238)},
        {new Byte (BibleBooks.Amos),"AMOS",new Long(26089)},
        {new Byte (BibleBooks.Obadiah),"OBADJA",new Long(3906) },
        {new Byte (BibleBooks.Jonah),"JONA",new Long(7566) },
        {new Byte (BibleBooks.Micah),"MICHA",new Long(19057) },
        {new Byte (BibleBooks.Nahum),"NAHUM",new Long(7648) },
        {new Byte (BibleBooks.Habakkuk),"HABAKUK",new Long(9144) },
        {new Byte (BibleBooks.Zephaniah),"ZEFANJA",new Long(9594) },
        {new Byte (BibleBooks.Haggai),"HAGGAI",new Long(6871) },
        {new Byte (BibleBooks.Zechariah),"SACHARJA",new Long(38055) },
        {new Byte (BibleBooks.Malachi),"MALEACHI",new Long(10911) },

        // New Testament
        {new Byte (BibleBooks.Matthew),"MT",new Long(159047)},
        {new Byte (BibleBooks.Mark),"MK",new Long(94761)},
        {new Byte (BibleBooks.Luke),"LK",new Long(163309)},
        {new Byte (BibleBooks.John),"JOH",new Long(121562)},
        {new Byte (BibleBooks.Acts),"APG",new Long(152808) },
        {new Byte (BibleBooks.Romans),"ROEMER",new Long(68828)},
        {new Byte (BibleBooks.Corinthians1),"1_KOR",new Long(65039)},
        {new Byte (BibleBooks.Corinthians2),"2_KOR",new Long(41786)},
        {new Byte (BibleBooks.Galatians),"GAL",new Long(21927)},
        {new Byte (BibleBooks.Ephesians),"EPH",new Long(22264)},
        {new Byte (BibleBooks.Philippians),"PHIL",new Long(15834)},
        {new Byte (BibleBooks.Colossians),"KOL",new Long(14291)},
        {new Byte (BibleBooks.Thessalonians1),"1_THESS",new Long(13339)},
        {new Byte (BibleBooks.Thessalonians2),"2_THESS",new Long(7349)},
        {new Byte (BibleBooks.Timothy1),"1_TIM",new Long(17428)},
        {new Byte (BibleBooks.Timothy2),"2_TIM",new Long(12247)},
        {new Byte (BibleBooks.Titus),"TITUS",new Long(7365)},
        {new Byte (BibleBooks.Philemon),"PHLM",new Long(3096)},
        {new Byte (BibleBooks.Hebrews),"HEBR",new Long(47434)},
        {new Byte (BibleBooks.James),"JAK",new Long(16199)},
        {new Byte (BibleBooks.Peter1),"1_PETRUS",new Long(17912)},
        {new Byte (BibleBooks.Peter2),"2_PETRUS",new Long(10788)},
        {new Byte (BibleBooks.John1),"1_JOH",new Long(16466)},
        {new Byte (BibleBooks.John2),"2_JOH",new Long(2024)},
        {new Byte (BibleBooks.John3),"3_JOH",new Long(2064)},
        {new Byte (BibleBooks.Jude),"JUDAS",new Long(4546)},
        {new Byte (BibleBooks.Revelation),"OFFB",new Long(73374)},

        // Aphocryhen
        {new Byte (BibleBooks.Judith),"JUDIT",new Long(52843)},
        {new Byte (BibleBooks.Wisdom),"WEISHEIT",new Long(73176)},
        {new Byte (BibleBooks.Tobit),"TOBIAS",new Long(42821)},
        {new Byte (BibleBooks.Sirach),"SIRACH",new Long(190608)},
        {new Byte (BibleBooks.Baruch),"BARUCH",new Long(30917)},
        {new Byte (BibleBooks.Maccabees1),"1_MAKK",new Long(144724)},
        {new Byte (BibleBooks.Maccabees2),"2_MAKK",new Long(106216)},
        {new Byte (BibleBooks.DanielAddOn),"S_ZU_DAN",new Long(23767)},
        {new Byte (BibleBooks.EstherAddOn),"S_ZU_EST",new Long(16265)},
        {new Byte (BibleBooks.Manasses),"MANASSE",new Long(2164)},
    };
}
