package de.jbible.service.JBibleTextProvider;

import java.io.*;
import java.util.*;

/**
 * Index List for the JBible file format.
 * Represents an optimize stored index list
 * in memory.
 */

public class IndexList
{
    /**
     * Create and load a new list from stream.
     *
     * The read data can either stored as byte stream
     * or as int[]. The constructor convers all data from
     * byte stream to int[] if the blConvert flag is set. Use then
     * the getFromIndex() method. Otherwise use the getFromPosition ()
     * method.
     *
     * @param in The input stream to read from.
     * @param decompress Convert the read data to a int array
     *		in constrcutor.
     * @exception IOException An reading error could occure.
     *
     * @see #getFromPosition
     * @see #getFromIndex
     */
	public IndexList(DataInputStream in,boolean decompress)
    	throws IOException
	{
    	int size = in.readInt();
        _count = in.readInt();
        int storage = in.readShort();

        // Create buffer
        _buffer = new byte[size];
        in.read(_buffer);


        // set internal vars:

        // ident storage flags:
		int filter =	SINGLE_BYTE | DOUBLE_BYTE|
        				TRIPPLE_BYTE| QUAD_BYTE;

        switch (storage & filter)
        {
        	case SINGLE_BYTE :
            	_maxByteCount = 1;
                break;
            case DOUBLE_BYTE :
            	_maxByteCount = 2;
                break;
            case TRIPPLE_BYTE :
            	_maxByteCount = 3;
                break;
            case QUAD_BYTE :
            	_maxByteCount = 4;
                break;
            default :
            	throw new IOException ("Illegal max byte count id.");
        }

        // identify follow bits:
        _followFlag[0] = ((storage & IndexList.FOLLOW_FLAG_BIT1) == IndexList.FOLLOW_FLAG_BIT1);
        _followFlag[1] = ((storage & IndexList.FOLLOW_FLAG_BIT2) == IndexList.FOLLOW_FLAG_BIT2);
        _followFlag[2] = ((storage & IndexList.FOLLOW_FLAG_BIT3) == IndexList.FOLLOW_FLAG_BIT3);
        _followFlag[3] = false;

        if (decompress)
        	doDecompress ();

    }

    /**
     * Decompress the whole array at once.
     *
     * The _buffer will eliminated within this function.
     */
    private void doDecompress ()
    {
        int data;
        int shift;
        int b,i,pos=0,j;

        _indexList = new int [_count];

		for (j=0;j<_count;j++)
        {
        	shift = 0;
            data = 0;
            for (i=0;i<_maxByteCount;i++)
            {
                // read byte
                b = _buffer[pos++];

                if (_followFlag [i])
                {
                    // add byte to data:
                    data |= (b & 0x7f) << shift;
                    if ((b & 0x80) != 0x80)
                        break; // no byte follows!
                    shift += 7;
                }
                else
                {
                    // add byte to data:
                    data |= (b & 0xff) << shift;
                    shift += 8;
                }
            }
            _indexList[j] = data;
		}

    }


    /**
     * Access a index value.
     *
     * @param index the index of the index value.
     * @return The value at the index.
	 */
	public int getFromIndex (int index)
    {

        return _indexList [index];
    }



    /**
     * Access a index value.
     *
     * @param postion The byte position of the requested entry.
     * @return The value at the index.
	 */
	public int getFromPosition (int position)
    {
     	return getFromPosition (new Position (position));
	}

    /**
     * Access a index value.
     *
     * @param postion The byte position of the requested entry.
     * @return The value at the index.
	 */
	public int getFromPosition (Position position)
    {
        int data = 0;
        int shift = 0;
        int b;

        for (int i=0;i<_maxByteCount;i++)
        {
            // read byte
            b = _buffer[position._position++];

            if (_followFlag [i])
            {
                // add byte to data:
                data |= (b & 0x7f) << shift;
                if ((b & 0x80) != 0x80)
                    break; // no byte follows!
                shift += 7;
            }
            else
            {
                // add byte to data:
                data |= (b & 0xff) << shift;
                shift += 8;
            }
        }

        // add data to list:
		return data;
    }



    /**
     * Access the entry count of the array.
     *
     * @return The entry count of the array
     */
	public int size ()
    {
    	return _count;
    }

    /**
     * Access the length of the internal byte array.
     *
     * This method can only be called, if this object
     * was created with decompress=false.
     *
     * @return The byte array length.
     */
	public int length ()
    {
    	return _buffer.length;
    }


    /**
     * The follow flag array for decoding.
     *
     * Each value corresponds to a byte
     * in the data value and specifiy if
     * this byte has a follow flag or not.
     */
	private boolean _followFlag[] = new boolean [4];

     /**
      * The maximum length of a data word.
      */
	private int _maxByteCount;

    /**
     * The data buffer.
     *
     * Cound be null.
     */
	private byte [] _buffer;

    /**
     * The decompressed index list.
     *
     * Cound be null.
     */
	private int [] _indexList;

    /**
     * The count of data entires.
     */
    private int _count;

	/**
     * Storing size define: Max 1 Byte.
     */
	public static final short SINGLE_BYTE = 0;

    /**
     * Storing size define: Max 2 Byte.
     */
	public static final short DOUBLE_BYTE = 1;

    /**
     * Storing size define: Max 3 Byte.
     */
	public static final short TRIPPLE_BYTE = 2;

    /**
     * Storing size define: Max 4 Byte.
     */
	public static final short QUAD_BYTE = 3;


    /**
     * Storing size define: The last bit of the first byte specifies that
     * a second byte follows.
     */
	public static final short FOLLOW_FLAG_BIT1 = 4;

    /**
     * Storing size define: The last bit of the second byte specifies that
     * a third byte follows.
     */
	public static final short FOLLOW_FLAG_BIT2 = 8;

    /**
     * Storing size define: The last bit of the third byte specifies that
     * a forth byte follows.
     */
	public static final short FOLLOW_FLAG_BIT3 = 16;



}
