package de.jbible.service.JBibleTextProvider;

import java.io.*;
import java.util.*;
import java.util.zip.*;
import de.jbible.core.*;
import de.jbible.bible.*;

/**
 * A bible text in the JBible file format.
 */
public class JBibleText implements BibleText
{

	/**
     * Create and load a new object from file.
     */
	public JBibleText(InputStream inp )
    	throws IOException
	{
    	// Create ZIP stream:
/*        ZipInputStream zip = new ZipInputStream (inp);
        ZipEntry ent = zip.getNextEntry();*/

        DataInputStream in = new DataInputStream (inp);

    	// Read Header:
        _version = in.readInt();

        if (_version != 1 /*|| !"text".equals(ent.getName())*/)
        	throw new IOException ("Wrong file version!");

        _title = in.readInt();
        _copyright = in.readInt();
        _copyrightIdentifier = in.readInt();
        _locale = in.readInt();

        // read index lists:
        _stringList = new StringArray (in);
        _sentenceList = new IndexList (in,false);
        _wordList = new IndexList (in,false);
		_chapterIndexList = new IndexList (in,false);
  		_bookIndexList = new IndexList (in,true);
        _bookNameList = new IndexList (in,true);
		IndexList bookIdList = new IndexList (in,true);

        // create book array from book index list:
        // create also the book id mapping table:
        int size = bookIdList.size();
        _bookIdArray = new byte [size];
        _bookIdMapping = new byte [256];
        byte value;

        for (int i=0;i<size;i++)
        {
        	value = (byte)bookIdList.getFromIndex (i);
            _bookIdArray [i] = value;
            _bookIdMapping [value+128] = (byte)i;
        }
	}

	public String getTitle()
	{
		return getSentence (_title);
	}

	public String getCopyright()
	{
		return getSentence (_copyright);
	}

	public int getCopyrightIdentifier()
	{
		return _copyrightIdentifier;
	}

	public String getLocale()
	{
		return getSentence (_locale);
	}

    /**
     * Returns the book title
     *
     * @param bookId The id of the book as defined in BibleBooks.
     * @return The title of the book as string.
     */
	public String getBookTitle (byte bookId)
	{
    	return getSentence (_bookNameList.getFromIndex(mapBookId (bookId)));
	}
    /**
     * Create a sentence.
     *
     * @param start The start position on the sentence.
     * @return The sentence as string.
     */
	private String getSentence (int start)
    {
    	// restore sentence:
        int i;
        int data;
        StringBuffer result = new StringBuffer ();
        String s;

        Position pos = new Position (start);

        do
		{
        	data = _sentenceList.getFromPosition(pos);

            // test separator:
            if (data == 0)
            	break;
            s = _stringList.get(data);
            addString (result,s);

		} while (true);

        return result.toString();
    }

	/**
     * Compute the right whitespace between the existing
     * buffer and the data to add.
     *
     * @param buf The buffer to add the string.
     * @param str The string to add.
     */
	private void addString (StringBuffer buf,String str)
    {
    	int size = buf.length();
        char c;
        if (size != 0)
        {
        	if (str.length() == 1)
            {
                c = str.charAt(0);
                switch (c)
                {
                    case ',' :
                    case '.' :
                    case ';' :
                    case ':' :
                    // not complete!!
                        break;
                    default :
                        buf.append(" ");
                        break;
                }
            }
            else
            {
                buf.append(" ");
            }
        }

      	buf.append(str);
    }

   /**
     * Access to all books.
     *
     * This method returns a list of all books of this
     * translation. The array contains book identifier as
     * defined in BibleBooks.
     *
     *
     * @return The book identifier array.
     * @see de.jbible.bible.BibleBooks
     */
	public byte [] getBooks ()
    {

        // create new array and copy:
        byte[] result = new byte [_bookIdArray.length];
        System.arraycopy(_bookIdArray,0,result,0,result.length);

		return result;
    }

    /**
     * Map a book identifier to the internal index:
     *
     * @param id The id of the book
     * @return The internal index of this book.
     */
	public byte mapBookId (byte id)
    {
    	return _bookIdMapping [id+128];
    }

    /**
     * Get the chapter count of a book.
     *
     * @param book The book identifier as defined in BibleBooks
     * @return The count of chapters of this book.
     * @see BibleBooks
     */
	public int getChapterCount (byte book)
    {
		// The chapter counts must be calculated
        // from the chapter array.

        // First: we must know the book position:
		byte bookPos =  mapBookId (book);

        // get chapter pos:
        int chapterPos = _bookIndexList.getFromIndex(bookPos);
        int chapterEndPos;

        // get the reading end position:
        if (bookPos < _bookIndexList.size()-1)
        	chapterEndPos = _bookIndexList.getFromIndex(bookPos+1);
        else
			chapterEndPos = _chapterIndexList.length();

        Position ip = new Position (chapterPos);
        int count = 0;

        while (ip._position < chapterEndPos)
        {
            // read chapter:
            _chapterIndexList.getFromPosition (ip);
			count++;
        }
		return count;
    }

    /**
     * Get the vers count of a chapter.
     *
     * @param book The book identifier as defined in BibleBooks
     * @param chapter The chapter of the book the vers count is requested for.
     * @return The count of chapters of this book.
     * @see BibleBooks
     */
	public int getVersCount (byte book,int chapter)
    {
		// The chapter counts must be calculated
        // from the chapter array.

        // First: we must know the book position:
		byte bookPos =mapBookId (book);

        // get chapter pos:
        int chapterPos = _bookIndexList.getFromIndex(bookPos);
        int chapterEndPos;

        Position ip = new Position (chapterPos);
        int i,startPos=0;

        for (i=-1;i<chapter;i++)
	        startPos = _chapterIndexList.getFromPosition (ip);

        // read the beginning


        // we have now the start pos within the word list
        // and must count the verses.

        int endPos;

        // find end pos.
        if (ip._position < _chapterIndexList.length()-1)
        	endPos = _chapterIndexList.getFromPosition (ip);
        else
        	endPos = _wordList.length();

        int count = 1,value;
        ip = new Position (startPos);
        while (ip._position < endPos)
        {
         	value= _wordList.getFromPosition (ip);

            // count the vers separators:
            if (value == 0)
	            count ++;
        }

 		return count;

    }


    /**
     * Access a vers.
     *
     * You get the vers as String array. To create the whole
     * sentence, use the TextCompositor class.
     *
     * @see TextCompositor
     * @param loc The requested vers.
     * @return The vers array.
     * @exception IndexOutOfBounds The location does not exists in
     *    this bible.
     */
	public List getVersStrings (byte book,int chapter,int vers)
    {
		// The chapter counts must be calculated
        // from the chapter array.

        // First: we must know the book position:
		byte bookPos = mapBookId (book);

        // get chapter pos:
        int chapterPos = _bookIndexList.getFromIndex(bookPos);
        int chapterEndPos;

        Position ip = new Position (chapterPos);
        int i,startPos=0;

        for (i=-1;i<chapter;i++)
	        startPos = _chapterIndexList.getFromPosition (ip);

        // read the beginning


        int value;
        ip = new Position (startPos);

        for (i=0;i<vers;/*conditional, see below*/)
        {
        	value= _wordList.getFromPosition (ip);

            // count the vers separators:
            if (value == 0)
	            i++;
        }

        // Create vers:
        List result = new ArrayList ();
        String s;

        int len = _wordList.length();
		// read words from the word list:
        while (ip._position < len)
        {
        	value= _wordList.getFromPosition (ip);
            if (value == 0)
            	break;

            // get the string
            s = _stringList.get(value);
            result.add(s);
        }

		return result;

    }





    /**
     * The version number of the read bible file.
     */
    private int _version;

    /**
     * The title of this translation.
     * A reference to the _sentence index list.
     */
    private int _title;

    /**
     * The copyright of this translation.
     * A reference to the _sentence index list.
     */
    private int _copyright;

    /**
     * The copyright identifer,
     * defined in BibleText
     *
     * @see de.jbible.bible.BibleText#COPYRIGHT_PUBLICDOMAIN
     * @see de.jbible.bible.BibleText#COPYRIGHT_BUYWARE
     * @see de.jbible.bible.BibleText#COPYRIGHT_SHAREWARE
     * @see de.jbible.bible.BibleText#COPYRIGHT_WEBPUBLISH
     * @see de.jbible.bible.BibleText#COPYRIGHT_TOFRIENDS
     */
    private int _copyrightIdentifier;

    /**
     * The locale of this translation.
     * A reference to the _sentence index list.
     */
    private int _locale;


    /**
     * Book index list.
     * Referes to the chapter index list.
     */
    private IndexList _bookIndexList;

    /**
     * Book name list.
     * Referes to the sentence list.
     */
    private IndexList _bookNameList;

    /**
     * Book identifier list.
     * Referes to the defines in BibleBooks.
     *
     * @see de.jbible.bible.BibleBooks
     */
    private byte[] _bookIdArray;

    /**
     * Mapping table from book id to book position.
     */
   	private byte [] _bookIdMapping;

    /**
     * The chapter index list.
     *
     * Referes to the word list.
     */
    private IndexList _chapterIndexList;

    /**
     * The word list of all bible words in order.
     *
     * Refers to the string list. Verses are separated
     * by 0.
     */
    private IndexList _wordList;

    /**
     * List of several sentence (Book title, copyright,...).
     * Referes to teh string list. Sentence are separated
     * by 0.
     */
    private IndexList _sentenceList;

    /**
     * The string pool.
     */
    private StringArray _stringList;
}
