package de.jbible.service.MainWindow;

import de.jbible.tool.swinglang.*;
import de.jbible.core.*;
import de.jbible.*;
import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import java.net.*;
import java.util.*;
/**
 * This is a main window service.
 *
 * It's implemented with Swing classes and implemets the
 * {@link de.jbible.core.WindowService WindowService}.
 */
public class MainWindow extends DefaultService implements WindowService, WindowListener,
    LocaleChangeListener
{

    /**
     * Register this service.
     *
     * @param context The service context.
     */
    public void init (ServiceContext context)
    {
        super.init(context);
        context.registerTitle("Main Window Service");
        context.registerResourceBundle("de.jbible.service.MainWindow.MainWindowResourceBundle");
    }


    /**
     * Create and show the main window.
     *
     */
    public void start ()
    {

// init frame :
       _frame = new LangJFrame (_context);
       _frame.addWindowListener (this);
       _frame.setTitle(Main.APP_TITLE);
        URL url = ClassLoader.getSystemResource (
            "de.jbible/service/MainWindow/frameicon.gif");
        if (url != null)
        {
            _frame.setIconImage(
                Toolkit.getDefaultToolkit().getImage(url));
        }

// put gui together:
       _frame.setContentPane (_desktop);
       _frame.setJMenuBar (_menuBar);

// Create menu

        _bibleMenu = new LangJMenu (_context);
        addMenu (_bibleMenu,"MW_bible");
        _fileMenu = new LangJMenu (_context);
        addMenu (_fileMenu,"MW_file");
        _editMenu = new LangJMenu (_context);
        addMenu (_editMenu,"MW_edit");
        _toolMenu = new LangJMenu (_context);
        addMenu (_toolMenu,"MW_tools");
        _viewMenu = new LangJMenu (_context);
        addMenu (_viewMenu,"MW_view");
        _helpMenu = new LangJMenu (_context);
        addMenu (_helpMenu,"MW_help");

        addMenuItem (_fileMenu,"MW_exit",new ActionListener () {
            public void actionPerformed (ActionEvent ev)
            { doExit();}},
            "de/jbible/service/MainWindow/exit.gif",
            "MW_exit_desc",
            KeyStroke.getKeyStroke('X',Event.CTRL_MASK)
            );
        _fileMenu.insertSeparator (0);

        // set default PLAF:
        String plaf = _context.getProperty (PLAF);

        if (plaf == null)
        {
            plaf = UIManager.getLookAndFeel().getClass().getName();
        }

// create Look-and-Feel menu
        UIManager.LookAndFeelInfo []info = UIManager.getInstalledLookAndFeels();
        JRadioButtonMenuItem itm=null;
        LangJMenu lookAndFeelMenu = new LangJMenu(_context);

        int i;
        for (i=0;i<info.length;i++)
        {
            // Create new Menu Item:
            itm = new JRadioButtonMenuItem ();
            itm.setText (info[i].getName ());
            itm.setActionCommand (info[i].getClassName());
            itm.addActionListener (new ActionListener ()
            { public void actionPerformed (ActionEvent e)
                { setPLAF (e);} });


            _lookAndFeelButtonGroup.add (itm);

            if (info[i].getClassName().equals (plaf))
                itm.setSelected (true);
            else
                itm.setSelected (false);

            lookAndFeelMenu.add (itm);
        }
        lookAndFeelMenu.setText ("MW_plaf");
        lookAndFeelMenu.setToolTipText ("MW_plaf_desc");
        lookAndFeelMenu.setMnemonic ('L');
        lookAndFeelMenu.setIcon (loadImageIcon (
            "de/jbible/service/MainWindow/plaf.gif"));
        _viewMenu.add (lookAndFeelMenu);

// Create Locale menu:
        LangJMenu localeMenu = new LangJMenu (_context);
        localeMenu.setText ("MW_language");
        localeMenu.setToolTipText ("MW_language_desc");
        localeMenu.setIcon (loadImageIcon (
            "de/jbible/service/MainWindow/language.gif"));

        // create subitems:
        Locale [] locale = _context.getSupportedLocales();
        Locale actual = _context.getLocale ();
        for (i=0;i<locale.length;i++)
        {
            itm = new JRadioButtonMenuItem (locale[i].getDisplayName());
            itm.setActionCommand (locale[i].toString());
            itm.addActionListener (new ActionListener ()
                {   public void actionPerformed (ActionEvent e)
                    { setLanguage (e); }});
            if (actual.equals(locale[i]))
                itm.setSelected (true);
            else
                itm.setSelected (false);

            localeMenu.add (itm);
            _languageButtonGroup.add (itm);
        }
        _viewMenu.add (new JSeparator());
        _viewMenu.add (localeMenu);

// Center in screen:
        centerScreen (_frame,new Dimension (800,600));

// set preselected plaf:
        try
        {
            if (plaf != null && !"".equals (plaf))
            {
                UIManager.setLookAndFeel (plaf);
                updateUI();
            }
        }
        catch (Exception ex)
        {
            LoggingManager.error ("Cant set default PLAF:" + plaf,ex,
                this);
        }

// make visible
        _frame.setVisible (true);

    }


    /**
     * Proceed exit command from the user.
     */
    private void doExit ()
    {
        _context.exit(0);
    }

    /**
     * Set the PLAF on user request.
     *
     * @param e The event that causes this method call
     */
    private void setPLAF (ActionEvent e)
    {
        String className = null;
        try
        {
            className = e.getActionCommand();
            UIManager.setLookAndFeel (className );

            _context.setProperty (PLAF,className);

            // be sure that all instances will have the same
            // PLAF:
            _context.setLocale (_context.getLocale());

        }
        catch (Exception x)
        {
            LoggingManager.error ("Change PLAF: "+className,
                x,this);
        }
    }

    /**
     * Set the language on user request.
     *
     * @param e The event that causes this method call
     */
    private void setLanguage (ActionEvent e)
    {
        Locale l[] = _context.getSupportedLocales();
        for (int j=0;j<l.length;j++)
        {
            if (l[j].toString().equals (e.getActionCommand()))
            {
                _context.setLocale (l[j]);
                break;
            }
        }
    }

    /**
     * Center a frame to the screen.
     *
     * The size can be presetted or the actual frame size can be used.
     *
     * @param frame Specifies the frame that should be centered
     * @param preset Specifies a preset dimension. If this is null, the dimension
     *              of the frame is used.
     */
    private void centerScreen (Window frame,Dimension preset)
    {
        Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
        Dimension frameSize;

        if (preset == null)
            preset = frame.getSize();

        if (preset.height > screenSize.height)
            preset.height = screenSize.height;
        if (preset.width > screenSize.width)
            preset.width = screenSize.width;

        // set new pos:
        frame.setBounds((screenSize.width - preset.width) / 2,
                         (screenSize.height - preset.height) / 2,
                         preset.width,
                         preset.height);
    }


    /**
     * Load image icon from url string.
     *
     * @param urlString The URL to load from.
     * @return The ImageIcon Object if found, or null
     */
    private ImageIcon loadImageIcon (String urlString)
    {
        if (urlString != null)
        {
            URL url = ClassLoader.getSystemResource (urlString);
            if (url != null)
                return new ImageIcon (url);
        }
        return null;
    }

    /**
     * Create one menu.
     *
     * The method creates a menu and add it to the menu bar.
     * The strings and the mnemonics came from the resource,
     * a icon is set and  ToolBar is created.
     *
     * @param menu The menu object to add and initialize.
     * @paran name The name of the menu item.
     */
    private void addMenu (JMenu menu,String name)
    {
        menu.setText (name);
        menu.setToolTipText (name +"_desc");
        menu.setIcon (loadImageIcon ("de/jbible/service/MainWindow/"
            + name.substring(3)+".gif"));
        menu.setMnemonic (name.charAt(0));

        _menuBar.add (menu);
    }

    /**
     * Add a menu item to the menu.
     *
     * The menu item is created and added. The method try to load
     * a icon for the item.
     *
     * @param addTo The menu where to add the new created item.
     * @param name The name (title) of the item. It's a key to the
     *  resource bundle.
     * @param action The action listener of this item.
     * @param iconName The URL of the icon that should be used. Could be null.
     * @param description A description text to this icon. It's a key to the
     *  resource bundle. Could be null.
     * @param key A accelerator for this item. Could be null.
     */
    private void addMenuItem (JMenu addTo,String name,ActionListener action,
        String iconName,String description,KeyStroke key)
    {
        // create item
        LangJMenuItem item = new LangJMenuItem (_context);
        item.setText (name);

        item.setIcon (loadImageIcon (iconName));
        item.setMnemonic (name.charAt(0));
        item.addActionListener (action);

        if (description != null)
            item.setToolTipText (description);

        if (key != null)
            item.setAccelerator (key);
        addTo.insert (item,0);

    }

    /**
     * This method is called before the application exits.
     *
     * You should write back data here.
     */
    public void exit()
    {
        _frame.setVisible (false);
    }


    /**
     * Registers an action.
     *
     * The action is added to the user interface, e.G. to the menu and/or
     * a tool bar. You must specify a action group, so that the Window
     * can put the action to the right position in the menu structure.
     *
     * @param name The name of the action This name will be displayed,
     * it must be language dependend!
     * @param group The group, see Group definitions below.
     * @param action The action that should be performed.
     * @param iconName Name of the icon for this action, must
     *  be relative to the classpath, with "/", e.G. "bible/bla/x.gif"
     * @param description A description of this action (key to the
     *  resource bundle), used for the status bar.
     *
     * @throws IllegalGroupException The group identifier is illegal or not
     *  supported by this service.
     *
     * @return true, if action was registered fine, else false
     */
    public boolean registerAction (String name,int group,
        ActionListener action,String iconName,String description)
        throws IllegalGroupException
    {
        JMenu menu = null;
        switch (group)
        {
            case WindowService.GRP_BIBLE :
                menu = this._bibleMenu;
                break;
            case WindowService.GRP_EDIT :
                menu = this._editMenu;
                break;
            case WindowService.GRP_FILE :
                menu = this._fileMenu;
                break;
            case WindowService.GRP_HELP :
                menu = this._helpMenu;
                break;
            case WindowService.GRP_TOOLS :
                menu = this._toolMenu;
                break;
            default :
                throw new IllegalGroupException (group);
        }

        // add menu:
        addMenuItem (menu,name,action,iconName,description,null);

        return true;
    }

    /**
     * Creates a frame within the desktop.
     *
     * You must specify the name of the frame. What kind of object is
     * created depends on the GUI component. What you get is a component.
     * This might be the first container or a JInternalFrame ro whatever.
     * To get rid of the frame call the frame.setVisible (false) method
     * and remove it from its parent with :
     * frame.getParent().remove (frame)
     *
     * @param frameName The name of the frame.
     * @param content The content of the frame.
     * @param iconName Name of the icon for this action, must
     *  be relative to the classpath, with "/", e.G. "bible/bla/x.gif"
     * @param resizeable It this window resizeable?
     *
     * @return You get the frame (or something like a frame) here.
     *
     * @see javax.swing.JComponent#getParent
     * @see java.awt.Container#remove
     */
    public Container createFrame (String frameName,Container content,
        String iconName,boolean resizeable)
    {
        JInternalFrame frame = new JInternalFrame (frameName,resizeable,
            true,true,true);
        frame.setFrameIcon(loadImageIcon (iconName));
        frame.setContentPane (content);
        frame.pack();
        _desktop.add (frame,JLayeredPane.DEFAULT_LAYER);
        _desktop.moveToFront(frame);

        return frame;
    }

    /**
     * Creates a modal dialog on the desktop.
     *
     * You must specify the name of the dialog. What kind of object is
     * created depends on the GUI component. What you get is a component.
     * This might be the first container or a JInternalFrame ro whatever.
     * To get rid of the dialog call the dialog.setVisible (false) method
     * and remove it from its parent with :
     * dialog.getParent().remove (dialog)
     *
     * @param dialogName The name of the dialog.
     * @param content The content of the dialog.
     * @param iconName Name of the icon for this action, must
     *  be relative to the classpath, with "/", e.G. "bible/bla/x.gif"
     *
     * @return You get the dialog (or something like a dialog) here.
     *
     * @see javax.swing.JComponent#getParent
     * @see java.awt.Container#remove
     */
    public Window createDialog (String frameName,Container content,
        String iconName)
    {
        JDialog dialog = new LangJDialog (_frame,frameName,_context);
        dialog.setContentPane (content);
        centerScreen (dialog,new Dimension (200,300));

        return dialog;

    }

    /**
     * This method is called if the locale has changed.
     *
     * The service should react on it and change all
     * its language dependend stuff.
     * <b>Important:</b> Dont call setLocale to the ServiceManager
     * or ServiceContext within this message!
     *
     * @see TheServiceManager#setLocale
     * @param newLocale The new locale
     */
    public void localeChanged(Locale newLocale)
    {
        updateUI ();
    }

    /**
     * Helper method that call updateUI for all windows.
     */
    private void updateUI ()
    {
        try
        {
            // update GUI:
            SwingUtilities.updateComponentTreeUI(_frame);

            Window [] w = _frame.getOwnedWindows();
            Window wi;
            for (int i=0;i<w.length;i++)
            {
                wi = w[i];
                SwingUtilities.updateComponentTreeUI(wi);
                // Translate window itself:
                if (wi instanceof WindowUpdateInterface)
                {
                    ((WindowUpdateInterface)wi).updateUI();
                }
            }
            // translate the frame itself:
            _frame.updateUI();
        }
        catch (Exception x)
        {
            LoggingManager.error("Error in updateUI:",x,this);
        }

    }


    /**
     * Exit Application on closing main window.
     *
     * Method comes from the WindowListener interface.
     *
     * @param param1 The WindowEvent, is unused here.
     */
    public void windowClosing(WindowEvent parm1)
    {
        _context.exit(0);
    }


   /**
     * Creates a logging window with optional progress bar.
     *
     * This method creates a frame with a logging area and optional
     * a progress bar. The idea is to have a facility that logs actions
     * that takes several minutes. The progress could be visualized with the
     * progress bar and with messages, warnings and errors. Its depends
     * on the returned window how the differences are visualzed.
     *
     * The user of this facility must call the finish method to show that
     * his action has finished. Then a close- button should be displayed.
     *
     * @return The created facility.
     */

    public ProgressWindowInterface createProgressWindow (String actionTitle,
        String iconName)
    {
        JPanel p = new ProgressPanel (_context);
        Container cont = createFrame (actionTitle,p,iconName,true);
        cont.setSize(300,200);
        cont.setVisible(true);

        return (ProgressWindowInterface)p;

    }


    /** Unused WindowLister method. */
    public void windowActivated(WindowEvent parm1) {}
    /** Unused WindowLister method. */
    public void windowClosed(WindowEvent parm1){}
    /** Unused WindowLister method. */
    public void windowDeactivated(WindowEvent parm1) {}
    /** Unused WindowLister method. */
    public void windowDeiconified(WindowEvent parm1) {}
    /** Unused WindowLister method. */
    public void windowIconified(WindowEvent parm1) {}
    /** Unused WindowLister method. */
    public void windowOpened(WindowEvent parm1) {}


// Member variables:

// GUI elements:
    /** The frame of the window. */
    private LangJFrame _frame;
    /** The MDI Area. */
    private JDesktopPane _desktop = new JDesktopPane ();
    /** The button group for the PLAF selection. */
    private ButtonGroup _lookAndFeelButtonGroup = new ButtonGroup();
    /** Button group for language selection menu. */
    private ButtonGroup _languageButtonGroup = new ButtonGroup();

// Menu:
    /** The menu bar. */
    private JMenuBar _menuBar = new JMenuBar ();
    /** The Bible menu. */
    private LangJMenu _bibleMenu;
    /** The file menu. */
    private LangJMenu _fileMenu;
    /** The import submenu. */
//    private LangJMenu _importMenu = null;
    /** The edit menu. */
    private LangJMenu _editMenu;
    /** The tools menu. */
    private LangJMenu _toolMenu;
    /** The view menu. */
    private LangJMenu _viewMenu;
    /** The help menu. */
    private LangJMenu _helpMenu;


    private static final String PLAF = "PlugableLookAndFeel";

}
