package de.jbible.tool.bibleimport;

import java.util.*;

/**
 * This visitor collects all data that
 * are neccessary to write the bible file.
 */

public class BibleFileCreatorVisitor implements Visitor
{

	/**
     * See interface description for explanation.
     *
     * @param pool The string pool where all string of the
     * 		visited tree are stored in.
     * @see Visitor
     */
	public BibleFileCreatorVisitor(StringPool pool)
	{
    	_pool = pool;
    }

	/**
     * See interface description for explanation.
     *
     * @see Visitor
     */
	public int visitBook(Book book)
	{

    	// Create book list entries:
        _bookId.addEntry(book.getBookId());

        // get the name of the book.
        Sentence bookName = book.getName();

        // store this sentence in the sentence list:
        // the start index is the size;
        // because that is the next adress we can write to.
        _bookName.addEntry(_sentence.size());

        // write sentence
        // All strings are normalized from the previous
        // run of the AnalyseVisitor.
        Iterator iter = bookName.getWords();
        while (iter.hasNext())
        	_sentence.add(((Word)iter.next()).getWord());
        _sentence.add (null);

		// write index to chapter list:
        _book.addEntry(_chapter.size());

		return Visitor.CONTINUE;

    }

	/**
     * See interface description for explanation.
     *
     * @see Visitor
     */
	public int visitChapter(Chapter chapter)
	{
    	// write the index of the staring vers into the chapter list.
        _chapter.addEntry(_word.size()+1); // +1: do not point to the terminator

		return Visitor.CONTINUE;
	}

	/**
     * See interface description for explanation.
     *
     * @see Visitor
     */
	public int visitVers(Vers vers)
	{
        // add separator to word list:

        // dont add a 0 to the start:
        if (_word.size() != 0)
	        _word.add(null);

		return Visitor.CONTINUE;
	}

	/**
     * See interface description for explanation.
     *
     * @see Visitor
     */
	public int visitSentence(Sentence sentence)
	{
		return Visitor.CONTINUE;
	}

	/**
     * See interface description for explanation.
     *
     * @see Visitor
     */
	public int visitWord(Word word)
	{
		return Visitor.CONTINUE;
	}

    /** Not used jet. */
  	public int visitBibleWord (Word word)
    {
    	// store string reference to word list.
        _word.add(word.getWord());

    	return Visitor.NO_CHILDS;
    }

	/**
     * See interface description for explanation.
     *
     * @see Visitor     */
	public int visitMetaInfo(MetaInfo metainfo)
	{
		return Visitor.CONTINUE;
	}

    /**
     * Add a sentence to the sentence list.
     *
     * This method adds a sentence in a string, not
     * a sentence object.
     *
     * @param toAdd The sentence to add
     * @return The start index in the sentence array.
     */
	public int addSentence (String toAdd)
    {
		int result = _sentence.size();
        StringTokenizer tokenizer = new StringTokenizer (toAdd," ");
        String token;

        while (tokenizer.hasMoreTokens())
        {
        	// get token
        	token = tokenizer.nextToken();
            // normalize token (add to string pool, if not in)
 			token = _pool.add(token);
            // add to sentence list:
            _sentence.add(token);
        }

        // terminate sentence:
        _sentence.add(null);

        return result;
    }

    /**
     * Access the book index list.
     *
     * @return The book index list.
     */
	public IndexListWriter getBookIndexList ()
    {
    	return _book;
    }

    /**
     * Access the book id list.
     *
     * @return The book id list.
     */
	public IndexListWriter getBookIdList ()
    {
    	return _bookId;
    }

    /**
     * Access the book name list.
     *
     * @return The book name list.
     */
	public IndexListWriter getBookNameList ()
    {
    	return _bookName;
    }

    /**
     * Access the chapter index list.
     *
     * @return The chapter index list.
     */
	public IndexListWriter getChapterIndexList ()
    {
    	return _chapter;
    }


    /**
     * Access the word list.
     *
     * @return The word list.
     */
    public IndexListWriter getWordList ()
    {
    	return createIndexList (_word);
    }

    /**
     * Access the sentence list.
     *
     * @return The sentence list.
     */
	public IndexListWriter getSentenceList ()
    {
    	return createIndexList (_sentence);
    }

    /**
     * Access the string pool.
     *
     * @return The string pool.
     */
    public StringPool getStringPool ()
    {
    	_pool.sort();
        return _pool;
    }

    /**
     * Create a DefaultIndexList which's entries
     * refers to the string pool from a string
     * reference array.
     *
     * All strings must allready be part of the string pool.
     *
     * @param in The input list.
     * @return The generated DefaultIndexList.
     */
	public IndexListWriter createIndexList (List in)
    {
        // the pool must be sorted:
        _pool.sort ();

    	IndexListWriter result = new IndexListWriter ();

        // iterate over the whole list:
        Iterator iter = in.iterator();
        String s;
        int order;

        while (iter.hasNext())
        {
        	s = (String)iter.next();

            if (s == null)
            	order = 0;
            else
				order = _pool.getOrder (s)+1;

            result.addEntry(order);
        }

        return result;
    }

    /**
     * Get rid of the string pool.
     *
     * This method is for memory saving.
     */
	public void looseStringPool ()
    {
     	_pool = null;
    }

    /**
     * Get rid of the sentence array.
     *
     * This method is for memory saving.
     */
	public void looseSentences ()
    {
     	_sentence = null;
    }

    /**
     * Get rid of the string pool.
     *
     * This method is for memory saving.
     */
	public void looseWordList ()
    {
     	_word = null;
    }

    /**
     * Get rid of the chapter list.
     *
     * This method is for memory saving.
     */
	public void looseChapterList ()
    {
     	_pool = null;
    }
    /**
     * Book index list, referes to chapter list.
     */
    private IndexListWriter _book = new IndexListWriter ();
    /**
     * Book define list, referes to consts in de.jbible.bible.BibleBooks.
     * @see de.jbible.bible.BibleBooks
     */
    private IndexListWriter _bookId = new IndexListWriter ();

    /**
     * Book headline list. Referes to the vers list.
     */
    private IndexListWriter _bookName = new IndexListWriter ();

    /**
     * Chapter list, referes to the vers list.
     */
    private IndexListWriter _chapter = new IndexListWriter ();


    /**
     * Word list, refers to the string list.
     *
     * We can build a IndexList after the whole bible is read.
     */
    private List _word = new ArrayList ();

    /**
     * Sentence list, refers to the string list.
     *
     * We can build a IndexList after the whole bible is read.
     */
    private List _sentence = new ArrayList ();

    /**
     * The string pool.
     */
    private StringPool _pool;

}
