package de.jbible.tool.bibleimport;

import java.util.*;
import java.util.zip.*;
import java.io.*;
import de.jbible.core.*;
import de.jbible.bible.*;
import de.jbible.service.JBibleTextProvider.*;

/**
 * The Bible fileformat import writer.
 *
 * This class writes the bible in the internal file format.
 * The import itself is handled in the run method, in an extra thread.
 *
 */
public class BibleImportWriter implements Runnable
{

	/**
     * Create a new object. Specify the reader object and a
     * progress window which logs the progress.
     *
     * @param reader The import reader object.
     * @param progress The Progress window object.
     * @param context The actual service context, used for translation.
     */
    public BibleImportWriter(BibleImportReader reader,
    	ProgressWindowInterface progress, ServiceContext context)
    {
    	_reader = reader;
        _progress = progress;
        _context = context;
    }

    /**
     * Do writing. This can be done in an extra thread.
     */
    public void run ()
    {

		Book book=null;
        Iterator chapterIter;
        Chapter chapter;
        Iterator versIter;
        Vers vers;
        Iterator wordIter;
        Word word;
        List bookList = new LinkedList ();
        DataOutputStream out = null;
		ZipOutputStream zip = null;


		AnalyseVisitor analyseVisitor = new AnalyseVisitor ();
        BibleFileCreatorVisitor createVisitor =
        	new BibleFileCreatorVisitor  (analyseVisitor.getStringPool());

        _progress.logMessage((String)_context.translate("IDS_readBible"),
        	_progress.MESSAGE);

        // fake progress bar
        float bookProgress = ((float)70) / (float)BibleBooks.bookCount;
        float progress = 0;


        // writing algorithm:
        // read book per book:
        //	- normalze all strings and add them into the string pool
        //  - Create book, chapter and word list.
        // Write lists back

       	try
        {
	        do
    	    {

             	book = (Book)_reader.readBook ();

                if (book != null)
                {
					book.visit(analyseVisitor);
                    book.visit(createVisitor);
    	            _progress.logMessage(" "+
                    	book.getName(),
                        _progress.MESSAGE);
                }
                progress += bookProgress;
				_progress.setProgressValue((int)progress);

	        }
            while (book != null);
//            while (false);

            // write now:
            String fileName =
            	ServiceManager.getStaticProperty("BIBLEPATH")+
				File.separatorChar +
                _reader.getFileName()+
                ".jbible";

            // add sentence of header
            int title = createVisitor.addSentence (_reader.getTitle());
            int copy = createVisitor.addSentence (_reader.getCopyright());
            int locale = createVisitor.addSentence (_reader.getLocale());

            // compute all indexes so that they
            // refer to the compressed byte stream
            IndexListWriter sentence = createVisitor.getSentenceList();
            IndexListWriter wordList = createVisitor.getWordList();
            StringPool pool = createVisitor.getStringPool();
            sentence.prepareWriting ();

        	// create header
            HeaderWriter header = new HeaderWriter ();
            header.setVersion(1); // the actual version is 1
            header.setCopyrightIdentifier(_reader.getCopyrightIdentifier());
            header.setCopyright (sentence.mapIndex(copy));
            header.setTitle(sentence.mapIndex(title));
            header.setLocale (sentence.mapIndex(locale));

			_progress.setProgressValue(70);

	        _progress.logMessage((String)_context.translate("IDS_writeBible"),
            	_progress.MESSAGE);

            FileOutputStream fout =  new FileOutputStream (fileName);

/*            zip = new ZipOutputStream (fout);
            zip.setLevel(9);
            zip.putNextEntry(new ZipEntry ("text"));
  */

            // open file to write:
//            out = new DataOutputStream (new FileOutputStream (fileName));
            out = new DataOutputStream (fout);


            // write header:
            header.write(out);

            // write strings:
            pool.write (out);
            pool = null;
            createVisitor.looseStringPool ();
			_progress.setProgressValue(75);


            // write sentence index list:
            sentence.write (out);

            createVisitor.getBookNameList().refToBytes(sentence);
            createVisitor.looseSentences ();
            sentence = null;
			_progress.setProgressValue(80);

            // write word index list:
            wordList.write(out);
   			_progress.setProgressValue(85);

            // write chapter index list:
            IndexListWriter writer;
            writer = createVisitor.getChapterIndexList();
            writer.refToBytes (wordList);
            writer.write (out);
            createVisitor.looseWordList ();
            wordList = null;


   			_progress.setProgressValue(90);


            // write book index list:
            writer = createVisitor.getBookIndexList();
            writer.refToBytes (createVisitor.getChapterIndexList());
            writer.write (out);
            createVisitor.looseChapterList ();
   			_progress.setProgressValue(95);


            // write book name list:
            writer = createVisitor.getBookNameList();
            writer.write (out);

            // write book Id list:
            writer = createVisitor.getBookIdList();
            writer.write (out);

            // debugging:
			//_progress.logMessage(""+header,_progress.MESSAGE);

	        _progress.logMessage((String)_context.translate("IDS_readyWriting"),
            	_progress.MESSAGE);

        }
        catch (Exception e)
        {
            System.err.println(e.toString());
            LoggingManager.error("Error while reading word.",e,this);
        }
        finally
        {
        	if (out != null)
            {
            	try
                {
                    out.flush();

                    if (zip != null)
                        zip.closeEntry();

                	out.close();
                }
                catch (IOException ex)
                {
					LoggingManager.error("Error while closing file:",
                    	ex,this);
                }
            }
        }

        // statistics:
/*        String stat = pool.toString();
        stat += "\nBooks: " + bookCount;
        stat += "\nChapters: " + chapterCount;
        stat += "\nVerses: " + versCount;
        stat += "\nWords: " + wordCount;*/
//		_progress.logMessage(analyseVisitor.toString(),_progress.MESSAGE);


        _progress.finished();
    }


    /** The facility where to log the progress. */
    private ProgressWindowInterface _progress;

    /** The reader objet. */
    private BibleImportReader _reader;


    /** Service Context. Used for translation. */
    private ServiceContext _context;

}


