package de.jbible.tool.bibleimport;

import de.jbible.bible.*;
import java.util.*;


/**
 * Represents a book.
 *
 * This object is only used for import filters.
 */

public class Book extends MetaInfoContainer
{

	/** Not allowed. */
	private  Book(){ super (null);}

   /**
     * Create a new Book object with given number and chapters and meta infos.
     *
     * The chapters are a linked list. The list will be destroyed
     * and the objects will be added to the internal list.
     *
     * @param number The number of this book.
     * @param chapters One or more chapters, linked internally together.
     */
    public Book (String number,Chapter chapters,MetaInfo meta)
    {
    	super (meta);
		_id = Byte.parseByte(number);

        Debug._bookNr = _id;

        addChapters (chapters);
    }

    /**
     * Add chapter(s) to this object.
     *
     * @param chapters The meta info(s) to add. It can be one
     *		or more, with the next ref internally linked together.
     *		Each element on its own is added to the internal list.
     *		The next param will be set to null.
     */
    public void addChapters (Chapter chapters)
    {
    	if (chapters == null)
        	return;

    	if (_chapters == null)
        	_chapters = new LinkedList ();

    	Chapter next;
    	do
        {
        	_chapters.add(chapters);
            next = chapters._next;
            chapters._next = null;
            chapters = next;

        }
        while (chapters != null);
    }

	/**
     * Access the chapter list.
     *
     * @return The iterator over all chapters, can be null.
     */
	public Iterator getChapters ()
    {
    	if (_chapters == null)
        	return null;

    	return _chapters.iterator();
    }

    /**
     * Set the id of this book.
     *
     * This refers to the defines in BibleBooks
     *
     * @see de.jbible.bible.BibleBooks
     */
    public void setBookId (byte id)
    {
    	_id = id;
    }

    /**
     * Get the id of this book.
     *
     * This refers to the defines in BibleBooks
     *
     * @see de.jbible.bible.BibleBooks
     */
    public byte getBookId ()
    {
    	return _id;
    }

    /**
     * Debugging purpose.
     *
     * @return The name of this book. Its the english name,
     * the same as the define.
     */
    public String toString ()
    {
    	return "Book: "+Utilities.getBookName (_id);
    }

    /**
     * Visit the whole data tree.
     *
     * @param visitor The visitor object.
     */
	public int visit (Visitor visitor)
    {
		// visit this object
    	if (visitor.visitBook(this) != Visitor.NO_CHILDS)
		{
            // visit the MetaInformations:
            super.visit (visitor);


            // visit childs:
            if (_chapters != null)
            {
	            Iterator iter = _chapters.iterator();

            	while (iter.hasNext())
               		((Chapter)iter.next()).visit(visitor);
			}
        }
        return Visitor.CONTINUE;
    }

    /**
     * Set the name of the book.
     * This name is in the books original language.
     *
     * @param name The name of the book.
     */
	public void setName (Sentence name)
    {
    	_name = name;
    }


    /**
     * Access the name of the book.
     *
     * @return The name of this book.
     */
    public Sentence getName ()
    {
    	return _name;
    }

    /**
     * The id of this book.
     *
     * This refers to the defines in BibleBooks
     *
     * @see de.jbible.bible.BibleBooks
     */
    private byte _id;

    /** List of chapters. */
    private List _chapters ;

    /** Name of this book. */
    private Sentence _name;

}