package de.jbible.tool.bibleimport;

import java.io.*;
import java.util.*;
import de.jbible.core.*;
import de.jbible.service.JBibleTextProvider.*;

/**
 * This class contains an index list as used for the book-,
 * chapter-, vers-, and sentence index list.
 */

public class IndexListWriter
{
    /**
     * Add a new index to the list
     *
     * @param index The new index.
     */
	public void addEntry (int index)
    {
        _indexList.add (new Integer (index));
    }

    /**
     * Get the count of entires of the index array.
     *
     * @return The count of entries.
     */
	public int size ()
    {
    	if (_indexList != null)
	    	return _indexList.size();
        if (_mapping != null)
        	return _mapping.length;
        return 0;
    }

    /**
     * Access a entry.
     *
     * @param index The index the value is requested for.
     * @return The value from the requested index.
     */
	public int getValue (int index)
    {
    	return ((Integer)_indexList.get(index)).intValue();
    }


    /**
     * This method computes the optimized storage for this
     * index list.
     *
     * @return The optimized method. There are some defines for them
     * in {@link de.jbible.tool.bibleimport.Header Header }
     */
    public short computeOptimizeStorage ()
    {
    	// analye the array:
        int maxBitUsed = 1;
        int iCountTotal = _indexList.size();
        int i,j,value;
        int histo[] = new int[32]; // Histogram

        // identify the maximum used bits:
        for (i = 0;i<iCountTotal;i++)
        {
			value = ((Integer)_indexList.get(i)).intValue();
            for (j=0;j<32;j++)
            {
            	value = value >> 1;
            	if (value  == 0)
                {
                	if (maxBitUsed < j)
                    	maxBitUsed = j;

                    // add entry o histogram
                    histo [j]++;
                    break;
                }
            }
        }

    	// check multibyte storage:
        int k;
        int bitPos,usedBitCount,totalMem,lastUsedByte;

        // Variables for the best option:
        short bestStorage=0;
        int bestMemUsage=-1;

        int byteMax;// max count of used bytes.

        for (i=0;i<2;i++) // iterate for follow flag of byte 1
        {
        	for (j=0;j<2;j++) // iterate for follow flag of byte 2
            {
				for (k=0;k<2;k++) // iterate for follow flag of byte 2
                {
                    usedBitCount = 0;
                    totalMem = 0;
                    lastUsedByte=0;
                    byteMax = ((maxBitUsed+i+j+k)>>3)+1;
                    if (byteMax > 4)
                    	continue; // too much for this file format, ignore
                    for (bitPos=0;bitPos<32;bitPos++)
                    {
                    	// find the follow-bits:
                    	if (!((i == 1 && bitPos == 7) ||
                        	  (j == 1 && bitPos == 15) ||
                              (k == 1 && bitPos == 23)))
                        {
                        	// yes, this bit is used for information:

                            // calc the count of bytes that are neccessary
                            // to store all list entries with this ammount
                            // of bits:

                            if (histo[usedBitCount] != 0)
                            {
                                lastUsedByte = ((bitPos>>3)+1);

                                // a data value can only end on a byte
                                // with a switch bit:
                                if (lastUsedByte == 1 && i == 0)
                                	lastUsedByte++; // data value cant end here!
                                if (lastUsedByte == 2 && j == 0)
                                	lastUsedByte++; // data value cant end here!
                                if (lastUsedByte == 3 && k == 0)
                                	lastUsedByte++; // data value cant end here!
                                if (lastUsedByte > byteMax)
                                	lastUsedByte = byteMax;

                                totalMem += histo[usedBitCount] * lastUsedByte;
                            }
                        	usedBitCount++;
                        }
                    }

                    // print result:
                    /*System.out.print("Storing: ");
                    if (i==1)
                    	System.out.print("Follow bit flag 1 ");
                    if (j==1)
                    	System.out.print("Follow bit flag 2 ");
                    if (k==1)
                    	System.out.print("Follow bit flag 3 ");
                  	System.out.println(""+totalMem+" Bytes");
                    */


					// find best storage:
                    if (bestMemUsage == -1 || bestMemUsage > totalMem)
                    {
                    	bestMemUsage = totalMem;
                    	// the last found opportunity is the best:
						switch (lastUsedByte)
                        {
                        	case 1 :
                            	bestStorage = IndexList.SINGLE_BYTE;
                                break;
                        	case 2 :
                            	bestStorage = IndexList.DOUBLE_BYTE;
                                break;
                        	case 3 :
                            	bestStorage = IndexList.TRIPPLE_BYTE;
                                break;
                        	case 4 :
                            	bestStorage = IndexList.QUAD_BYTE;
                                break;

                        }

                        if (i == 1)
                        	bestStorage |= IndexList.FOLLOW_FLAG_BIT1;
                        if (j == 1)
                        	bestStorage |= IndexList.FOLLOW_FLAG_BIT2;
                        if (k == 1)
                        	bestStorage |= IndexList.FOLLOW_FLAG_BIT3;
                    }

                }
            }
        }
    	return bestStorage;


    }

    /** Debug only. */
	public String toString ()
    {
    	return "Optimize storage: "+toString (computeOptimizeStorage());
    }

    /**
     * Debug only.
     * Create a readable string from the optimized storage flag.
     *
     * @param optimizeStorage The storage flag.
     * @return A readable english string.
     */
    public static String toString (short optimizeStorage)
    {
    	String result;
        final short filter = IndexList.SINGLE_BYTE |
        					IndexList.DOUBLE_BYTE |
                            IndexList.TRIPPLE_BYTE |
                            IndexList.QUAD_BYTE;

    	switch (optimizeStorage & filter)
        {
        	case IndexList.SINGLE_BYTE :
            	result = "Single Byte";
                break;
        	case IndexList.DOUBLE_BYTE :
            	result = "Double Byte";
                break;
        	case IndexList.TRIPPLE_BYTE :
            	result = "Tripple Byte";
                break;
        	case IndexList.QUAD_BYTE :
            	result = "QUAD Byte";
                break;
            default :
            	result ="Unknown";
                break;
        }

        if ((optimizeStorage & IndexList.FOLLOW_FLAG_BIT1) != 0)
        	result += " Byte 1 has a Follow flag";
        if ((optimizeStorage & IndexList.FOLLOW_FLAG_BIT2) != 0)
        	result += " Byte 2 has a Follow flag";
        if ((optimizeStorage & IndexList.FOLLOW_FLAG_BIT3) != 0)
        	result += " Byte 3 has a Follow flag";
		return result;
    }

    /**
     * Write this object to stream.
     *
     * List format:<br>
     * byte Storage type, as defind in IndexList<br>
     * int  Size of Data part<br>
     * ...  Data<br>
     *
     * @param out The output stream to write to.
     * @exception IOException Writing error.
     */
	public void write (DataOutputStream out)
    	throws IOException
    {
    	if (_data == null)
        	prepareWriting ();

        out.writeInt(_data.length);
        out.writeInt(_mapping.length);
        out.writeShort(_storage);

        // write data:
        out.write(_data);
    }

    /**
     * Prepare this list for writing.
     *
     * The optimized representation is created
     * and a mapping list from an index to the byte position.
     * This method must be called before write is called!
     * This method can called only once!
     *
     * @exception Occures on writing error or illegal data.
     */
	public void prepareWriting ()
        throws IOException
    {
    	_storage = computeOptimizeStorage();

        // Create writing buffer:
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        DataOutputStream writer = new DataOutputStream(bos);
		// write data into buffer:

        // iterate over the list:
        Iterator iter = _indexList.iterator();
        int value;
        int maxByteCount;
        int filter =  IndexList.SINGLE_BYTE|
        					IndexList.DOUBLE_BYTE|
                            IndexList.TRIPPLE_BYTE|
                            IndexList.QUAD_BYTE;

        switch (_storage & filter)
        {
        	case IndexList.SINGLE_BYTE :
            	maxByteCount = 1;
                break;
            case IndexList.DOUBLE_BYTE :
            	maxByteCount = 2;
                break;
            case IndexList.TRIPPLE_BYTE :
            	maxByteCount = 3;
                break;
            case IndexList.QUAD_BYTE :
            	maxByteCount = 4;
                break;
            default :
            	throw new IOException ("Illegal max byte count id.");
        }

        // identify follow bits:
        boolean followFlag[] = new boolean [4];
        followFlag[0] = ((_storage & IndexList.FOLLOW_FLAG_BIT1) == IndexList.FOLLOW_FLAG_BIT1);
        followFlag[1] = ((_storage & IndexList.FOLLOW_FLAG_BIT2) == IndexList.FOLLOW_FLAG_BIT2);
        followFlag[2] = ((_storage & IndexList.FOLLOW_FLAG_BIT3) == IndexList.FOLLOW_FLAG_BIT3);
        followFlag[3] = false;
        int i,mapPos=0;

        _mapping = new int [_indexList.size()];

        while (iter.hasNext())
        {
        	value = ((Integer)iter.next()).intValue();

            // store mapping:

            _mapping[mapPos++] = bos.size();

            for (i=0;i<maxByteCount;i++)
            {
				// did we have a switch bit?
                if (followFlag [i])
                {
                	// yes, we have a follow flag.

                    // are information following?
                    if ((value & 0xffffff80) == 0)
                    {
                    	// no information following:
                        writer.writeByte(value & 0x7f);
                        break; // stop inner iteration
                    }
                    else
                    {
                    	// information following:
                        writer.writeByte((value & 0x7f)| 0x80);
                    }

                    // shift value:
                    value >>= 7;
                }
                else
                {
                	// no follow flag:
                    writer.writeByte(value & 0xff);
                    value >>= 8;
                }
            }
        }

        writer.flush();

        // Copy data:
        _data = bos.toByteArray();
        _indexList = null; // get rid of index list for memory saving.
    }


    /**
     * Map an index to byte pos in optimized representation.
     *
     * @param index The index to map.
     * @return The byte pos of this index.
     */
	public int mapIndex (int index)
    {
        return _mapping[index];
    }




    /**
     * Exchange all indexes to the byte positions of
     * the depended list.
     *
     * @param depend The list this list depends to.
     */
	public void refToBytes (IndexListWriter depend)
    {
        ListIterator iter = _indexList.listIterator();
        int val;

        while (iter.hasNext())
        {
            val = ((Integer)iter.next()).intValue();
            // translate:

            val = depend.mapIndex(val);

            // add to result list
            iter.set(new Integer (val));
        }
    }



    /**
     * The list that conains the indices in this implementation.
     */
	private List _indexList = new ArrayList ();


    /**
     * The optimized representation of this list.
     * It is created in prepareWriting
     * @see #repareWriting
     */
	private byte [] _data;

    /**
     * The mapping list from an index to the position
     * in the optimized representation.
     * It is created in prepareWriting
     * @see #repareWriting
     * @see #_data
     */
	private int [] _mapping;

    /**
     * The optimize storage.
     * It is created in prepareWriting
     * @see #repareWriting
     */
	private short _storage;

}

