package de.jbible.tool.bibleimport;

import de.jbible.bible.*;
import java.io.*;

/**
 * Contains meta informations that depend to a vers, chapter
 * book or word.
 *
 * The different types are defined as constants in this class.
 * The user data can be used for several informations within
 * the browsing process. These data are not be used at storage.
 */
public class MetaInfo implements Serializable
{
	/**
     * Create a new metainfo object.
     *
     * @param type The type of this object.
     * 		It must be one of the values defined in this class.
     * @param value The value of the meta info.
     *		This is the information itself.
     */
	public MetaInfo(int type,Sentence value)
	{
    	_type = type;
        _sentence = value;
	}

	/**
     * Create a new metainfo object.
     *
     * @param type The type of this object.
     * 		It must be one of the values defined in this class.
     * @param value The value of the meta info.
     *		This is the information itself.
     * @param next The next metainfo object of a meta info chain
     */
	public MetaInfo(int type,Sentence value,MetaInfo next)
	{
    	this (type,value);
        _next = next;
	}

	/**
     * Create a new metainfo object.
     *
     * @param type The type of this object.
     * 		It must be one of the values defined in this class.
     * @param value The value of the meta info.
     *		This is the information itself.
     * @param next The next metainfo object of a meta info chain
     * @param userData Misc user data
     */
	public MetaInfo(int type,Sentence value,MetaInfo next,
    	Object userData)
	{
    	this (type,value,next);
        _userData = userData;
	}

    /** Not allowed. */
    private MetaInfo () {}
                       
    /**
     * Access to the sentence of this information.
     *
     * A meta info can contains either a sentence, e.G.
     * a comment or a VersLink.
     *
     * @return The sentence or null, if this object contains
     * 		a VersLink.
     */
	public Sentence getSentence ()
    {
    	switch (_type)
        {
            case HEADLINE :
            case COMMENT :
            case EXPLANATION :
            case TEMP :
            	return _sentence;
            default :
            	return null;
        }
    }

    /**
     * Access to the vers link of this information.
     *
     * A meta info can contains either a sentence, e.G.
     * a comment or a VersLink.
     *
     * @return The sentence or null, if this object contains
     * 		a sentence.
     */
	public VersLink getVersLink ()
    {
    	switch (_type)
        {
            case PARALLEL :
            	return _link;
            default :
            	return null;
        }
    }

    public String toString ()
    {
    	switch (_type)
        {
            case PARALLEL :
            	return "Parallel: "+getVersLink();
            case HEADLINE :
            	return "Headline: "+getSentence();
            case COMMENT :
            	return "Comment: "+getSentence();
            case EXPLANATION :
            	return "Explanation: "+getSentence();
            case TEMP :
            	return "Temp: "+getSentence();
        }
		return "Unknown: "+_type;
    }

    /**
     * Visit the whole data tree.
     *
     * @param visitor The visitor object.
     */
	public int visit (Visitor visitor)
    {
		// visit this object
    	if(visitor.visitMetaInfo(this) != Visitor.NO_CHILDS)
		{
	        // visit childs:
			if (_sentence != null)
				_sentence.visit(visitor);
        }
        return Visitor.CONTINUE;
    }

    /**
     * Access the type of this meta info.
     *
     * @return The type. This is one of the defines of this class.
     *
     * @see #HEADLINE
     * @see #COMMENT
     * @see #EXPLANATION
     * @see #PARALLEL
     * @see #TEMP
     */
	public int getType ()
    {
    	return _type;
    }

    /**
     * The type of this meta info object.
     *
     * It must be one of the values defined in this class.
     */
	private int _type;

    /** The information, if it is a sentence.  */
    private Sentence _sentence;

    /** The information, if it is a vers link. */
	private VersLink _link;

    /**
     * The next meta info.
     *
     * This is used only for construction.
     * After the metaInfo list is added to a
     * {@link MetaInfoContainer MetaInfoContainer},
     * these pointers are set to null.
     */
    transient MetaInfo _next;

    /** The user data */
    private Object _userData;


    /** MetaInfo type: Headline. */
    public static final int HEADLINE = 1;
    /** MetaInfo type: word parallel. */
    public static final int PARALLEL = 2;
    /** MetaInfo type: comment. */
    public static final int COMMENT = 3;
    /** MetaInfo type: explanation. */
    public static final int EXPLANATION = 4;
    /** MetaInfo type: temporaray. */
    public static final int TEMP = 5;
}
