package de.jbible.tool.swinglang;

import javax.swing.*;
import java.util.*;
import de.jbible.core.*;

/**
 * A combobox model that deals with language idependance.
 *
 * It contains a list of the langage keys to the visible entries.
 * The displayed values are translated by the ServiceContext.
 * Use the methods of this class to fill the combo box and to request
 * data. All this functions are working on resource bundle keys.
 *
 * <p>
 * This model is used by LangJComboBox.
 *
 * @see JComboBox
 */
public class LangComboBoxModel extends DefaultComboBoxModel
{
    /**
     * Inner class that contains a key, translation pair.
     */
    class TranslationData
    {
        /**
         * Create new object with initialisation.
         *
         * @param k The key.
         * @param v The value.
         */
        public TranslationData (Object k,String v)
        {
            key = k;
            value = v;
        }

        /** The language key. */
        public Object key;
        /** The actual translation. */
        public String value;

        /**
         * Comparator, is equal if the key is equal.
         */
        public boolean equals (Object obj)
        {
            return (obj instanceof TranslationData && key != null && key.equals (((TranslationData)obj).key)) ||
                   (obj != null && obj.equals (key));
        }

    };

    /**
     * Default constructor is not allowd.
     */
    private LangComboBoxModel()
    {
    }

    /**
     * Create a new model.
     *
     * The ServiceContext is needed to translate the strings.
     *
     * @param context The context for translation.
     */
    public LangComboBoxModel(ServiceContext context)
    {
        _context = context;
    }

    /**
     * Add a new element as language key.
     *
     * @param anObject The object to add.
     */
    public synchronized void addElement(Object anObject)
    {
        // create new tanslation data object:
        TranslationData data = new TranslationData (anObject,
            translate (anObject.toString()));
        _objectList.add (data);

        // add to the iherited data model:
        super.addElement (data.value);
    }

    /**
     * Select a item.
     *
     * Object is a language key.
     *
     * @param The object as language key.
     */
    public synchronized void setLangSelectedItem(Object anObject)
    {
        // find object:
        int index = _objectList.indexOf (anObject);

        if (index != -1)
        {
            TranslationData data = (TranslationData)_objectList.get (index);

            // set the value in the derived data object:
            super.setSelectedItem (data.value);
        }
    }

    /**
     * Get the actual selected item.
     *
     * The return value is a language key.
     *
     * @return The selected item, language independend.
     */
    public synchronized Object getLangSelectedItem()
    {
        // request selection from derived object:
        String obj = (String)super.getSelectedItem ();

        // find the value in the translation data:
        Iterator iter = _objectList.iterator();

        TranslationData data;

        while (iter.hasNext())
        {
            data = (TranslationData)iter.next();
            if (data.value.equals (obj))
            {
                // found result!
                return data.key;
            }
        }

        // no item found!
        return null;
    }


    private String translate (String value)
    {
        if (_context == null)
            return value;
        return (String)_context.translate (value);
    }


    /**
     * Method is called if the GUI change.
     *
     * This could be a change of the PLAF or the language.
     * Its called from
     */
    public synchronized void updateUI ()
    {
        if (getSize() != 0)
        {
            // clear model list of super

            removeAllElements();

            // translate all items:
            Iterator iter = _objectList.iterator();

            TranslationData data;
            while (iter.hasNext())
            {
                data = (TranslationData)iter.next();
                data.value = translate (data.key.toString());

                // add to super model:
                super.addElement (data.value);
            }
        }
    }

    /** The List of objects, used as language keys. */
    private List _objectList = new ArrayList ();

    /** The service context which translate for us. */
    private transient ServiceContext _context = null;

}