{***********************
  Media-Manager
************************
 (c) 1996 by
 Olaf Panz
 Drosselgasse 4
 21436 Marschacht
***********************}
unit UMMPaintArea;

interface

uses   SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls,
  WinG32,
  UMMPictureBase,
  UMMAssign,
  UMMTools,
  Memory;

type

TMMPaintArea = class (TGraphicControl)

   private
      { Private-Deklarationen }
      PaintList : TList; { Liste aller WinG- Komponenten }

      // WinG Variable
      m_WinGDC : HDC; // WinGDC
      m_BackBuffer : HBITMAP;  // BackBuffer- Bitmap
      m_pBackBuffer : PCHAR;   // Zeiger auf BackBuffer Inhalt
      m_hOldBmp : HBITMAP; // Original WinG-Bmp

      m_Palette : TPalette; // Palettendaten
      m_hPal : HPALETTE; // WinG- Palette als Handle, fr GetPalette
      m_Width,m_Height,m_Width32 : Integer; // GRe des WinG-Objekts
      FAutoRepaint : Boolean; { Soll nach jeder nderung automatisch neu
                               gezeichnet werden? }
      m_InvalidRect : TRect; { Ungiltiger Bereich frs nchste Neuzeichnen }

      m_hAssign : TAssignHandle;
  protected

    // Palette laden, m_hPal und m_Palette init
    procedure InitPalette;

    function GetPalette: HPALETTE; override;

    // zur Fenstergre passenden BackBuffer erzeugen
    procedure CreateBackBuffer;

    // WinG-Backbuffer freigeben
    procedure DestroyBackBuffer;

    // Bereich neu zeichnen :
    procedure PaintArea (Rect : TRect);
    // Aufruf aus Child-Komponente:
    procedure PaintAreaFromChild (Rect : TRect);

  public
    { Public-Deklarationen }

    { Registrieren, wenn Komponenten hinzugefgt werden. TMMPicture und
      Ableitungen werden in die Paint-Liste eingetragen }
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;

    procedure Paint; override;
    procedure UpdateArea; { Ungltigen Bereich neu darstellen }

    constructor Create (AOwner : TComponent); override;
    destructor Destroy; override;{ Ungltigen Bereich neu darstellen }


    // Palette neu laden
    procedure NewPalette;

  published

    property Width;
    property Height;
    property AutoRepaint : Boolean read FAutoRepaint write FAutoRepaint default FALSE;

    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;


end;

implementation

constructor TMMPaintArea.Create (AOwner : TComponent);
begin
   { Achtung: whrend des inherited create wird die procedure
     Notification aufgerufen. Wenn vorher nicht die PaintList
     Created wird, knallts ! }
   m_InvalidRect.Top := -1;


   PaintList := TList.Create;
   Align := alClient;

   { Defaults herstellen }
   m_WinGDC := 0;
   m_BackBuffer := 0;
   m_pBackBuffer := Nil;
   m_Width := 0;
   m_Height :=  0;
   FAutoRepaint := FALSE;

   inherited Create (AOwner);

   { Verbindung zu Mediendaten herstellen }
   m_hAssign := MMAOpenMedia (MMGetForm (AOwner).Name);


   // Paletten Variable m_hPal und m_Palette init
   InitPalette;

end;

destructor TMMPaintArea.Destroy;
begin
   if m_WinGDC <> 0 then
   begin
        DestroyBackBuffer; // BAckbuffer freigeben
        DeleteDC (m_WinGDC); // WinG freigeben
   end;

   // Palette freigeben
   if m_hPal <> 0 then
      DeleteObject (m_hPal);


   inherited Destroy;

   { Verbindung zu Mediendaten schlieen }
   MMACloseMedia (m_hAssign);
   PaintList.Free;
end;

{ Registrieren, wenn Komponenten hinzugefgt werden. TMMPicture und
  Ableitungen werden in die Paint-Liste eingetragen }
procedure TMMPaintArea.Notification(AComponent: TComponent; Operation: TOperation);
var index : integer;
begin

     if Operation =  opInsert then
        begin
           if AComponent is TMMPictureBase then
              begin
                 PaintList.Add (AComponent);
                 (AComponent as TMMPictureBase).SetPaintFunc (Self.PaintAreaFromChild);
              end;
        end
     else                         begin
           index := PaintList.IndexOf (AComponent);
           if index <> -1 then
              begin
                 PaintList.Delete (index);
              end;
        end;
end;

procedure TMMPaintArea.Paint;
var Rect : TRect;
begin

   // WinG gltig machen:
   if m_WinGDC = 0 then begin
      // Aktuelle Gre bernehmen

      m_WinGDC := WinGCreateDC;
      if m_WinGDC = 0 then
         MessageDlg ('Creating of WinG-DC failed',mtError,[mbAbort],0)
      else begin
         CreateBackBuffer;
      end;
   end else begin
      if (m_Width <> Width) or (m_Height <> Height) then begin
        DestroyBackBuffer;
        CreateBackBuffer;
      end;
   end;

   // und neu zeichnen
   Rect := Canvas.ClipRect;
   Rect.bottom := Rect.bottom +1;
   PaintArea (Rect);

//   PaintArea (Canvas.ClipRect);
   // Ungltigen Breich lschen


   inherited Paint;
end;

// WinG-Backbuffer freigeben
procedure TMMPaintArea.DestroyBackBuffer;
begin
   if m_BackBuffer <> 0 then begin
      SelectObject (m_WinGDC,m_hOldBmp);
      DeleteObject (m_BackBuffer);
      m_BackBuffer := 0;
      m_pBackBuffer := Nil;
   end;
end;

// zur Fenstergre passenden BackBuffer erzeugen
procedure TMMPaintArea.CreateBackBuffer;
var Info : TBitmapInfoPal;
    i : Integer;
begin
     // Datenstruktur lschen
     memset (@(Info.Header),0,sizeof (Info.Header));
     m_Width := Width;
     m_Width32 := Norm32 (m_Width);
     m_Height := Height;

     Info.Header.biSize := sizeof (TBitmapInfoHeader);
     Info.Header.biWidth := m_Width;
     Info.Header.biHeight := m_Height;
     Info.Header.biPlanes := 1;
     Info.Header.biBitCount := 8;
     Info.Header.biCompression := BI_RGB;

     // Palette in Datenstruktur kopieren
     memcpy32 (@ (Info.Pal),@m_Palette,sizeof (Info.Pal));

     m_BackBuffer := WinGCreateBitmap (m_WinGDC,@Info,Nil);
     if m_BackBuffer = 0 then begin
        MessageDlg ('Creating of WinG BackBuffer failed',mtError,[mbAbort],0);
        m_pBackBuffer := Nil;
     end else begin
        m_pBackBuffer := WinGGetDIBPointer (m_BackBuffer,Nil);
        if m_pBackBuffer = Nil then
           MessageDlg ('Getting WinG-BackBuffer Pointer failed',mtError,[mbAbort],0)
        else begin
           m_hOldBmp := SelectObject (m_WinGDC,m_BackBuffer);

           // Buffer mit Hintergrundfarbe fllen:
           for i:= 0 to m_Height-1 do
              memset32 (m_pBackBuffer + i*m_Width32,0,m_Width32);

        end;
     end;

end;

// Palette laden, m_hPal und m_Palette init
procedure TMMPaintArea.InitPalette;
type TMyLogPalette = record
        palVersion : WORD;
        palNumEntries : WORD;
        palPal : TPalette;
     end;
var pal : TMyLogPalette;
begin

     // Paletten-Daten ermitteln
     MMAGetMedia (m_hAssign,1,@m_Palette,sizeof (m_Palette));  // Palette ist immer 1 !!
     // Palettenstruktur erzeugen
     pal.palVersion := $300;
     pal.palNumEntries := 256;
     memcpy32 (@(pal.palPal),@m_Palette,sizeof (pal.palPal));

     m_hPal := CreatePalette (TLogPalette (PLogPalette (@pal)^));


end;

function TMMPaintArea.GetPalette: HPALETTE;
begin

     if m_hPal <> 0 then begin

        GetPalette := m_hPal
     end else
        GetPalette := inherited GetPalette;

end;

// Palette neu laden, nur im Entwurfmode verwendet
procedure TMMPaintArea.NewPalette;
//var DC : HDC;
//var hWin : HWND;
begin
   // Palette freigeben
   if m_hPal <> 0 then begin
      DeleteObject (m_hPal);
      m_hPal := 0;
   end;

   // und neu erzeugen
   InitPalette;
   WinGSetDIBColorTable (m_WinGDC,0,255,@m_Palette);
   Invalidate;

end;

procedure TMMPaintArea.UpdateArea;
begin
     // Wenn ein ungltiger Bereich existiert:
   if m_InvalidRect.Top  <> -1 then
   begin
      // neu zeichnen
      PaintArea (m_InvalidRect);
      m_InvalidRect.Top := -1;

   end;
end;

// Bereich neu zeichnen :
procedure TMMPaintArea.PaintAreaFromChild (Rect : TRect);
var InvRect : TRect;
begin

   if FAutoRepaint = TRUE then begin
      // gleich neu zeichnen:
      PaintArea (Rect)
   end else begin
       // Rect an InvalidRect anfgen
       if m_InvalidRect.Top  <> -1 then begin
          UnionRect (InvRect,m_InvalidRect,Rect);
          m_InvalidRect := InvRect;
       end else begin
          m_InvalidRect := Rect;
       end;
   end;

   end;


// Bereich neu zeichnen :
procedure TMMPaintArea.PaintArea (Rect : TRect);
var x,y,b : Integer;
    PaintWidth,PaintHeight : Integer;
    p : PChar;
    pict : TMMPictureBase;
begin
   // Parameter an Gre anpassen
   if Rect.Top < 0 then Rect.Top := 0;
   if Rect.Left < 0 then Rect.Left := 0;

   if Rect.Bottom >= Height then Rect.Bottom := Height;
   if Rect.Right >= Width then Rect.Right := Width ;

//   if Rect.Bottom >= Height then Rect.Bottom := Height -1;
//   if Rect.Right >= Width then Rect.Right := Width -1;

   // zu zeichnenden Bereich mit Hintergrundfarbe fllen:
   x := Rect.Left;
   p := m_pBackBuffer + x;
   b := m_Height - Rect.Bottom;

   PaintWidth := Rect.Right - x;
   PaintHeight := Rect.Bottom - Rect.Top;

   if Rect.Top < Rect.Bottom then
   begin
      for y :=  b to (Height - Rect.Top -1) do
      begin
           memset (p + y *m_Width32, 0 ,PaintWidth);
      end;

      // ber alle TMMPicture - Elemente iterieren und zum Zeichnen aufrufen
      x := PaintList.Count;

      for y := 0 to x-1 do
      begin
         pict := TMMPictureBase (PaintList.Items[y]);
         pict.PaintEx (m_pBackBuffer,Rect,
                          m_hAssign,m_Width32,m_Height);
      end;

   end;

   // Palette auswhlen
   SelectPalette (Canvas.Handle,m_hPal,FALSE);
   RealizePalette (Canvas.Handle);

   // Backbuffer auf Screen:
   WinGBitBlt (Canvas.Handle,Rect.Left,Rect.Top,PaintWidth +1,
              PaintHeight +1,m_WinGDC,Rect.Left,Rect.Top);
end;

end.
