{***********************
  Media-Manager
************************
 (c) 1996 by
 Olaf Panz
 Drosselgasse 4
 21436 Marschacht
***********************}
unit UMMSoundEmitter;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  UAAudio,UMMAssign,UMMTools;

type
  TMMSoundEmitter = class(TComponent)
  private
    { Private-Deklarationen }
    m_Position : TVector3D;
    m_Orientation : TVector3D;

    m_EmitterDesc : AAEMITTERDESC;
    m_EmitterModel : TAAEmitterModel;
    m_Emitter : IAAudioEmitter;
    m_SoundIdentifier : Integer;

    // Spielen zur Entwurfszeit
    m_PlayLoop : Boolean;

    m_hAssign : TAssignHandle; // Zugriffshandle fr Mediendaten

  protected
    { Protected-Deklarationen }
    procedure SetPosition (position : TVector3D);
    procedure SetOrientation (orientation : TVector3D);
    procedure SetEmitterModel (EmitterModel : TAAEmitterModel);

    procedure SetSoundIdentifier (Ident : Integer);

    // MediaControl
    function GetControlMedia : DWORD;
    procedure SetControlMedia (Control: DWORD);
    function GetPlaybackPosition : single;
    function GetTotalTime : single;

    // EmitterDescription
    procedure SetSoundFile (SoundFile : string);
    function GetSoundFile : string;

    // Emitter-Objekt erzeugen/lschen
    procedure CreateEmitter;
    procedure DestroyEmitter;

    // Updateing-Fuktionen
    procedure UpdateOrientation;
    procedure UpdatePosition;
    procedure UpdateEmitterModel;

  public
    { Public-Deklarationen }
    constructor Create (AOwner : TComponent); override;
    destructor Destroy; override;

    property ControlMedia : DWORD read GetControlMedia write SetControlMedia;
    property PlaybackPosition : single read GetPlaybackPosition;
    property TotalTime : single read GetTotalTime;
    property SoundFile : string read GetSoundFile write SetSoundFile;

    // Spiel-Funktionen
    procedure Play;
    procedure Loop;
    procedure Pause;
    procedure Resume;
    procedure Stop;

    procedure SetPlayLoop (Value : Boolean);

  published
    { Published-Deklarationen }

     // Position
    property Position : TVector3D read m_Position write SetPosition;
    property Orientation : TVector3D read m_Orientation write SetOrientation;
    property EmitterModel : TAAEmitterModel read m_EmitterModel write SetEmitterModel;

    // Spielen zur Entwurfszeit
    property LoopWhileDeveloping : Boolean read m_PlayLoop write SetPlayLoop stored FALSE;

    // Identifier:
    property Sound : Integer read m_SoundIdentifier write SetSoundIdentifier;

  end;


implementation

procedure TMMSoundEmitter.SetPosition (position : TVector3D);
begin
     m_Position := position;
     UpdatePosition;
end;

procedure TMMSoundEmitter.SetOrientation (orientation : TVector3D);
begin

     m_Orientation := orientation;
     UpdateOrientation;
end;

procedure TMMSoundEmitter.SetEmitterModel (EmitterModel : TAAEmitterModel);
begin
     m_EmitterModel := EmitterModel;
     UpdateEmitterModel;
end;

procedure TMMSoundEmitter.UpdateOrientation;
var erg : HRESULT;
begin
     if m_Emitter <> Nil then
     begin

        erg := m_Emitter.SetOrientation (@orientation.Vector);
        if erg <> S_OK then
          raise RSXException.Create ('UpdateOrientation',erg);
     end;
end;

procedure TMMSoundEmitter.UpdatePosition;
var erg : HRESULT;
begin
     if m_Emitter <> Nil then
     begin

        erg := m_Emitter.SetPosition (@position.Vector);
        if erg <> S_OK then
           raise RSXException.Create ('UpdatePosition',erg);
     end
end;

procedure TMMSoundEmitter.UpdateEmitterModel;
var erg : HRESULT;
begin
     if m_Emitter <> Nil then
     begin
        erg := m_Emitter.SetModel (@EmitterModel.EmitterModel);
        if erg <> S_OK then
             raise RSXException.Create ('UpdateEmitterModel',erg);
     end;
end;



// MediaControl
function TMMSoundEmitter.GetControlMedia : DWORD;
var control : DWORD;
    erg : HRESULT;
begin
     if m_Emitter <> Nil then
     begin
        erg := m_Emitter.QueryMediaState (@control,Nil,Nil);
        if erg <> S_OK then
           raise RSXException.Create ('GetControlMedia',erg);
        GetControlMedia := control;
     end else begin
        GetControlMedia := -1;
     end;
end;

procedure TMMSoundEmitter.SetControlMedia (Control: DWORD);
var erg : HRESULT;
begin
     if m_Emitter <> Nil then
     begin
        erg := m_Emitter.ControlMedia (Control);
        if erg <> S_OK then
           raise RSXException.Create ('SetControlMedia',erg);
     end;
end;

function TMMSoundEmitter.GetPlaybackPosition : single;
var pos : single;
    erg : HRESULT;
begin
     if m_Emitter <> Nil then
     begin
        erg := m_Emitter.QueryMediaState (Nil,@pos,Nil);
        if erg <> S_OK then
           raise RSXException.Create ('GetPlaybackPosition',erg);
        GetPlaybackPosition := pos;
     end else begin
        GetPlaybackPosition := 0.0;
     end;
end;

function TMMSoundEmitter.GetTotalTime : single;
var len : single;
    erg : HRESULT;
begin
     if m_Emitter <> Nil then
     begin
        erg := m_Emitter.QueryMediaState (Nil,Nil,@len);
        if erg <> S_OK then
           raise RSXException.Create ('GetTotalTime',erg);
        GetTotalTime := len;
     end else begin
        GetTotalTime := 0.0;
     end;
end;


// EmitterDescription
procedure TMMSoundEmitter.SetSoundFile (SoundFile : string);
begin
     if SoundFile <> '' then
     begin
        strpcopy (m_EmitterDesc.szFilename,SoundFile);
        DestroyEmitter;
        CreateEmitter;
     end;
end;

function TMMSoundEmitter.GetSoundFile : string;
begin
     GetSoundFile := m_EmitterDesc.szFilename;
end;


// Spiel-Funktionen
procedure TMMSoundEmitter.Play;
begin
     ControlMedia := AA_PLAYSINGLE;
end;

procedure TMMSoundEmitter.Loop;
begin
     ControlMedia := AA_PLAYLOOP;
end;

procedure TMMSoundEmitter.Pause;
begin
     ControlMedia := AA_PAUSE;
end;

procedure TMMSoundEmitter.Resume;
begin
     ControlMedia := AA_RESUME;
end;

procedure TMMSoundEmitter.Stop;
begin
     ControlMedia := AA_STOP;
end;


{ Public-Deklarationen }
constructor TMMSoundEmitter.Create (AOwner : TComponent);
begin
   // Defaults:
   m_Emitter := NIL;

   // Subobjekte erzeugen
   m_Position := TVector3D.Create (UpdatePosition);
   m_Orientation := TVector3D.Create (UpdateOrientation);
//   m_Velocity := TVector3D.Create;
   m_EmitterModel := TAAEmitterModel.Create (UpdateEmitterModel);

   m_Position.x := 0.0;
   m_Position.y := 0.0;
   m_Position.z := 0.0;

   m_Orientation.x := 0.0;
   m_Orientation.y := 0.0;
   m_Orientation.z := 1.0;

   m_EmitterDesc.szFilename := '';


   m_EmitterModel.minBack   := 2.0;
   m_EmitterModel.minFront  := 2.0;
   m_EmitterModel.maxBack   := 3.0;
   m_EmitterModel.maxFront  := 10.0;
   m_EmitterModel.intensity := 0.5;

   inherited Create (AOwner);

   m_hAssign := MMAOpenMedia (MMGetForm (AOwner).Name);

   CreateEmitter;
end;

destructor TMMSoundEmitter.Destroy;
begin
     DestroyEmitter;

     inherited Destroy;

     // Medienzugriff schlieen
     MMACloseMedia (m_hAssign);

     // Subobjeke lschen
     m_Orientation.Free;
     m_Position.Free;
     m_EmitterModel.Free;
end;

procedure TMMSoundEmitter.CreateEmitter;
var erg : HRESULT;
begin
   if (RSXAudio <> Nil) and (FileExists (m_EmitterDesc.szFilename) = TRUE) then
   begin
      try
         // Emitter erzeugen
         erg := RSXAudio.CreateEmitter (@m_EmitterDesc,@m_Emitter,NIL);
         if erg <> S_OK then
            raise RSXException.Create ('CreateEmitter',erg);

         // Emitter-Parameter einstellen
         UpdatePosition;
         UpdateOrientation;
         UpdateEmitterModel;
      except;
      end;
   end;
end;

procedure TMMSoundEmitter.DestroyEmitter;
begin
   if m_Emitter <> Nil then
   begin
      Stop;
      m_Emitter.Release;
      m_Emitter := Nil;
   end;
end;

procedure TMMSoundEmitter.SetPlayLoop (Value : Boolean);
begin
   m_PlayLoop := Value;
   if Value = TRUE then
      Loop
   else
      Stop;
end;

procedure TMMSoundEmitter.SetSoundIdentifier (Ident : Integer);
begin
     // Merken:
     m_SoundIdentifier := Ident;

     // File ermitteln
     SoundFile := MMAGetMediaFile (m_hAssign,m_SoundIdentifier);
end;



end.
