/**
 * Copyright 2005-2008 Olaf Panz 
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 *
 * 		http://www.apache.org/licenses/LICENSE-2.0 
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 */
package de.olafpanz.translationtable;

import java.io.IOException;
import java.io.Writer;

/**
 * Filters unicodes out and write them as \\uXXXX.
 * 
 * @author Olaf Panz
 * 
 */
final class ASCIIFilterWriter extends Writer {
    /**
     * Next writer in pipe.
     */
    private final Writer writer;

    /**
     * Create new instance
     * 
     * @param writer
     *                Next writer in pipe.
     */
    public ASCIIFilterWriter(final Writer writer) {
	assert writer != null;
	this.writer = writer;
    }

    /**
     * Convert string to unicode. Do not allow '"' in result, escape it.
     * 
     * @param str
     *                String to convert.
     * @return ASCII-String
     */
    public static final String convert(final String str) {
	final StringBuilder buf = new StringBuilder();

	final int iMax = str.length();
	for (int i = 0; i < iMax; i++) {
	    final char ch = str.charAt(i);
	    if (ch > '\177' || ch == '"') {
		buf.append('\\');
		buf.append('u');
		final String s1 = Integer.toHexString(ch);
		final int l = 4 - s1.length();
		for (int i1 = 0; i1 < l; i1++) {
		    buf.append('0');
		}
		buf.append(s1);
	    } else if (ch == '\t') {
		buf.append("    ");
	    } else {
		buf.append(ch);
	    }
	}
	return buf.toString();
    }

    /**
     * Delegate to next in pipe.
     */
    @Override
    public void close() throws IOException {
	writer.close();
    }

    /**
     * Delegate to next in pipe.
     */
    @Override
    public void flush() throws IOException {
	writer.flush();

    }

    /**
     * Filter and Delegate to next in pipe.
     */
    @Override
    public void write(final char[] ac, final int i, final int j)
	    throws IOException {
	for (int k = 0; k < j; k++) {
	    if (ac[k] > '\177') {
		writer.write(92);
		writer.write(117);
		final String s1 = Integer.toHexString(ac[k]);
		final int l = 4 - s1.length();
		for (int i1 = 0; i1 < l; i1++) {
		    writer.write('0');
		}
		writer.write(s1);
	    } else if (ac[k] == '\t') {
		writer.write(' ');
		writer.write(' ');
		writer.write(' ');
		writer.write(' ');
	    } else {
		writer.write(ac[k]);
	    }
	}

    }

}
