/**
 * Copyright 2005-2008 Olaf Panz 
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 *
 * 		http://www.apache.org/licenses/LICENSE-2.0 
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 */
package de.olafpanz.translationtable;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.EnumSet;
import java.util.List;
import java.util.Locale;
import java.util.Set;
import java.util.TreeSet;

/**
 * This represents one translation table read from ots file.
 * 
 * @author Olaf Panz
 */
final class CodeGenerator {
    /**
     * File encoding.
     */
    private static final String ENCODING = "US-ASCII";

    /**
     * Create a translation table with all meta information, without data.
     * 
     * @param doc
     *                Document to generate code for.
     * @param param
     *                Defines how to generate code.
     */
    public CodeGenerator(final TranslationDocument doc,
	    final TranslationTableParameters param) {
	this.doc = doc;
	this.param = param;
    }

    /**
     * Document to create source for.
     */
    private final TranslationDocument doc;
    /**
     * Code creation parameters.
     */
    private final TranslationTableParameters param;

    /**
     * Create all necessary files.
     * 
     * 
     * @throws IOException
     *                 Error in file creation
     */
    public void createFiles() throws IOException {

	final Set<TranslationEntry> data = doc.getTranslations();

	if (data.isEmpty()) {
	    // no translation specified, do no generate anything
	    return;
	}
	final boolean hasRCP = doc.hasRCP();
	final boolean hasBundle = doc.hasBundle();
	final File resourceBundleTargetDir;
	final String interfaceName;
	if (hasBundle) {
	    resourceBundleTargetDir = new File(param.getSourceRoot(), doc
		    .getPackageName().replace('.', File.separatorChar));
	    if (!resourceBundleTargetDir.exists()) {
		if (!resourceBundleTargetDir.mkdirs()) {
		    throw new IOException("Path creation failed: '"
			    + resourceBundleTargetDir + "'.");
		}
	    }

	    switch (param.getJdk()) {
	    case JDK14:
		interfaceName = createKeyInterface(resourceBundleTargetDir);
		break;
	    case JDK15:
	    default:
		interfaceName = createEnumInterface(resourceBundleTargetDir);
		break;

	    }
	} else {
	    interfaceName = null;
	    resourceBundleTargetDir = null;
	}
	// create default resource for eclipse-rcp
	createPluginProperties(-1);

	for (int i = 0; i < doc.getLocales().size(); i++) {
	    if (hasBundle) {
		createResourceFile(resourceBundleTargetDir, i, interfaceName);
	    }
	    if (hasRCP) {
		createPluginProperties(i);
	    }
	}
	if (hasBundle) {
	    createManager(resourceBundleTargetDir, interfaceName);
	}

    }

    /**
     * This structure is filled with types in use. Filled by
     * {@link #createKeyInterface(File)}
     */
    private final Set<TranslationDataType> usedTypes = EnumSet
	    .noneOf(TranslationDataType.class);

    /**
     * Create the translation manager for the specified environment.
     * 
     * @param targetDir
     *                The target dir
     * @param interfaceName
     *                Name of interface that defines keys.
     * @throws IOException
     *                 Error in file creation
     * 
     */
    private final void createManager(final File targetDir,
	    final String interfaceName) throws IOException {
	PrintWriter writer = null;
	final String clsName = doc.getClassName() + "TranslationManager";

	try {

	    final File fName = new File(targetDir, clsName + ".java");
	    writer = new PrintWriter(new BufferedWriter(new ASCIIFilterWriter(
		    new OutputStreamWriter(new FileOutputStream(fName),
			    ENCODING))));

	    writer.println("package " + doc.getPackageName() + ";");
	    writer.println();

	    writer.println("import javax.swing.KeyStroke;");
	    writer.println("import java.util.Locale;");
	    writer.println("import java.util.ResourceBundle;");
	    switch (param.getEnvironment()) {
	    case Server:
		writer.println("import java.util.Map;");
		writer.println("import java.util.HashMap;");
		break;
	    case Client:
		break;
	    }

	    writer.println();

	    writer.println("/**");
	    writer.println(" * " + doc.getClassComment());
	    writer
		    .println(" * This manager manages access to translation table.");
	    writer.println(" */");
	    writer.println("public final class " + clsName);
	    writer.println("{");
	    writer.println();

	    createMapLocaleMethod(writer);

	    // member-variable part
	    switch (param.getEnvironment()) {
	    case Server:
		writer.println("\t/**");
		writer.println("\t * Maps from locale to translation table");
		writer.println("\t */");
		writer
			.println("\tprivate final static Map<Locale,ResourceBundle> bundleMap = new HashMap<Locale,ResourceBundle>();");
		break;
	    case Client:
		writer.println("\t/**");
		writer.println("\t * The translation table");
		writer.println("\t */");
		writer.println("\t private static ResourceBundle bundle;");
		writer.println();
		writer.println("\t/**");
		writer.println("\t * Last returned locale.");
		writer.println("\t */");
		writer.println("\t private static Locale currentLocale;");
		break;
	    }
	    writer.println();

	    // create clean method
	    writer.println("\t/**");
	    writer.println("\t * Clean all cached resources.");
	    writer.println("\t */");
	    writer.println("\tpublic static final void clean (){");

	    switch (param.getEnvironment()) {
	    case Server:

		writer.println("\t\tsynchronized (" + clsName + ".class){");
		writer.println("\t\t\tbundleMap.clear();");
		writer.print("\t\t\t");
		break;
	    case Client:
		writer.println("\t\tbundle = null;");
		writer.print("\t\t");
		break;
	    }

	    writer.println("ResourceBundle.clearCache ();");
	    switch (param.getEnvironment()) {
	    case Server:
		writer.println("\t\t}");
		break;
	    case Client:
		break;
	    }

	    writer.println("\t}");
	    writer.println();

	    // Create get bundle method with locale
	    writer.println("\t/**");
	    switch (param.getEnvironment()) {
	    case Server:
		writer
			.println("\t * @param locale Locale of requested bundle.");
		break;
	    case Client:
		break;
	    }
	    writer.println("\t * @return bundle for default locale.");
	    writer.println("\t */");
	    writer.print("\tprivate static final ");
	    switch (param.getEnvironment()) {
	    case Server:
		writer
			.println(" ResourceBundle getBundle (final Locale locale) {");
		writer.println("\t\tResourceBundle bundle = null;");
		writer.println("\t\tsynchronized (" + clsName + ".class){");
		writer.println("\t\t\tbundle = bundleMap.get (locale);");
		writer.println("\t\t\tif (bundle == null){");

		writer.println("\t\t\t\tbundle = ResourceBundle.getBundle (\""
			+ doc.getPackageName() + "." + doc.getClassName()
			+ "\",mapLocale(locale));");
		writer.println("\t\t\t\tbundleMap.put (locale,bundle);");
		writer.println("\t\t\t}");
		writer.println("\t\t}");

		break;
	    case Client:
		writer.println(" ResourceBundle getBundle () {");
		writer
			.println("\t\tif (bundle==null || !Locale.getDefault().equals (currentLocale)){");
		writer.println("\t\t\tbundle = ResourceBundle.getBundle (\""
			+ doc.getPackageName() + "." + doc.getClassName()
			+ "\",mapLocale(Locale.getDefault()));");
		writer.println("\t\t}");
		break;
	    }
	    writer.println("\t\treturn bundle;");
	    writer.println("\t}");

	    // Create access methods
	    switch (param.getJdk()) {
	    case JDK14:
		if (usedTypes.contains(TranslationDataType.Text)) {
		    create14AccessMethod(writer, "Text", "String");
		}
		if (usedTypes.contains(TranslationDataType.Tooltip)) {
		    create14AccessMethod(writer, "Tooltip", "String");
		}
		if (usedTypes.contains(TranslationDataType.Accelerator)) {
		    create14AccessMethod(writer, "Accelerator", "KeyStroke");
		}
		if (usedTypes.contains(TranslationDataType.Mnemonic)) {
		    create14AccessMethod(writer, "Mnemonic", "Character");
		}
		break;
	    case JDK15:
	    default:
		if (usedTypes.contains(TranslationDataType.Text)) {
		    create15AccessMethod(writer, TranslationDataType.Text,
			    "String", interfaceName);
		}

		if (usedTypes.contains(TranslationDataType.Tooltip)) {
		    create15AccessMethod(writer, TranslationDataType.Tooltip,
			    "String", interfaceName);
		}

		if (usedTypes.contains(TranslationDataType.Mnemonic)) {
		    create15AccessMethod(writer, TranslationDataType.Mnemonic,
			    "Character", interfaceName);
		}

		if (usedTypes.contains(TranslationDataType.Accelerator)) {
		    create15AccessMethod(writer,
			    TranslationDataType.Accelerator, "KeyStroke",
			    interfaceName);
		}

	    }
	    writer.println();
	    writer.println("}");

	} finally {
	    if (writer != null) {
		writer.close();
	    }
	}

    }

    /**
     * Create code that evaluates the right resource bundle.
     * 
     * Locale assignment algorithm:
     * 
     * (1) Search for exact match considering language, country, variant
     * <p>
     * (2) Search for exact match considering language, country
     * <p>
     * (3) Search for exact match considering language
     * <p>
     * (4) Use default locale
     * 
     * @param writer
     *                Write code to this writer
     */
    private final void createMapLocaleMethod(final PrintWriter writer) {

	final List<LocaleDef> sortedLocales = new ArrayList<LocaleDef>(doc
		.getLocales());
	Collections.sort(sortedLocales);
	final String supportedLocalesVarName = "SUPPORTED_LOCALES";
	writer.println("\t/**");
	writer.println("\t * Array of Supported Locales. ");

	writer.println("\t */");
	writer.println("\tpublic static final Locale []"
		+ supportedLocalesVarName + " = new Locale[]{");

	// create defines for supported doc.getLocales()
	for (final LocaleDef locDef : sortedLocales) {
	    final Locale loc = locDef.getLocale();
	    writer.print("\t\tnew Locale (\"" + loc.getLanguage() + "\"");
	    String str = loc.getCountry();
	    if (str != null && !str.isEmpty()) {
		writer.print(",\"" + str + "\"");
	    }
	    str = loc.getVariant();

	    if (str != null && !str.isEmpty()) {
		writer.print(",\"" + str + "\"");
	    }
	    writer.println("),");

	}
	writer.println("\t};");
	writer.println();
	// create method header
	writer.println("\t/**");
	writer
		.println("\t * Maps from any locale to locale existing in translation table.");
	writer.println("\t @return Is never null.");
	writer.println("\t @param locale Null is not allowed here");
	writer.println("\t */");
	writer
		.println("\tpublic static final Locale mapLocale (final Locale locale){");
	writer.println("\t\t// (1) check for exact match");
	writer.println("\t\tfor (Locale loc : " + supportedLocalesVarName
		+ ") {");
	writer
		.println("\t\t\tif (loc.getLanguage().equals (locale.getLanguage()) && loc.getCountry().equals (locale.getCountry()) && loc.getVariant().equals (locale.getVariant())){");
	writer.println("\t\t\t\treturn loc;");
	writer.println("\t\t\t}");
	writer.println("\t\t}");

	writer.println("\t\t// (2) check for matching language and country");
	writer.println("\t\tfor (Locale loc : " + supportedLocalesVarName
		+ ") {");
	writer
		.println("\t\t\tif (loc.getLanguage().equals (locale.getLanguage()) && loc.getCountry().equals (locale.getCountry()) ){");
	writer.println("\t\t\t\treturn loc;");
	writer.println("\t\t\t}");
	writer.println("\t\t}");

	writer.println("\t\t// (3) check for matching language only");
	writer.println("\t\tfor (Locale loc : " + supportedLocalesVarName
		+ ") {");
	writer
		.println("\t\t\tif (loc.getLanguage().equals (locale.getLanguage())){");
	writer.println("\t\t\t\treturn loc;");
	writer.println("\t\t\t}");
	writer.println("\t\t}");

	writer.println("\t\t// (4) Use default language");
	writer.println("\t\treturn " + supportedLocalesVarName + "["
		+ doc.getDefaultLocale() + "];");
	writer.println("\t}");
    }

    /**
     * Create an access method for param.getJdk() 1.5
     * 
     * @param writer
     *                Writer to write code to
     * @param type
     *                Type of requested element
     * @param dataType
     *                Java data type of element
     * @param interfaceName
     *                Name of enum-interface
     */
    private void create15AccessMethod(final PrintWriter writer,
	    final TranslationDataType type, final String dataType,
	    final String interfaceName) {
	writer.println();
	writer.println("\t/**");
	writer.println("\t * Return " + type + " object.");
	writer
		.println("\t * @param key The key to requested resource. Null is not allowed.");

	switch (param.getEnvironment()) {
	case Server:
	    writer
		    .println("\t * @param locale Locale in that the resource is requested.");
	    break;
	case Client:
	    break;
	}
	writer.println("\t * @return Is never null.");

	writer.println("\t */");
	writer.print("\tpublic static final ");
	// switch (param.getEnvironment()) {
	// case Server:
	// writer.print("synchronized ");
	// break;
	// case Client:
	// break;
	// }
	writer.print(dataType + " get (final " + interfaceName + "."
		+ type.getPrefix() + " key");

	switch (param.getEnvironment()) {
	case Server:
	    writer.print(",final Locale locale");
	    break;
	case Client:
	    break;
	}

	writer.println(") {");

	writer.print("\t\treturn (" + dataType + ")getBundle (");
	switch (param.getEnvironment()) {
	case Server:
	    writer.print("locale");
	    break;
	case Client:
	    break;
	}

	writer.println(").getObject (key.toString());");
	writer.println("\t}");

    }

    /**
     * Create an access method for param.getJdk() 1.4
     * 
     * @param writer
     *                Writer to write source to.
     * @param objectType
     *                Name of object type, is used to create method name
     * @param dataType
     *                Name of java data type
     */
    private void create14AccessMethod(final PrintWriter writer,
	    final String objectType, final String dataType) {
	writer.println();
	writer.println("\t/**");
	writer.println("\t * Return " + objectType + " object.");
	writer
		.println("\t * @param key The key to requested resource. Null is not allowed.");
	switch (param.getEnvironment()) {
	case Server:
	    writer
		    .println("\t * @param locale Locale in that the resource is requested.");
	    break;
	case Client:
	    break;
	}
	writer.println("\t * @return Might be null, if key does not exist.");

	writer.println("\t */");
	writer.print("\tpublic static final ");
	// switch (param.getEnvironment()) {
	// case Server:
	// writer.print("synchronized ");
	// break;
	// case Client:
	// break;
	// }
	writer.print(dataType + " get" + objectType + " (final String key");

	switch (param.getEnvironment()) {
	case Server:
	    writer.print(",final Locale locale");
	    break;
	case Client:
	    break;
	}
	writer.println(") {");

	writer.print("\t\treturn (" + dataType + ")getBundle (");
	switch (param.getEnvironment()) {
	case Server:
	    writer.print("locale");
	    break;
	case Client:
	    break;
	}

	writer.println(").getObject (key);");

	writer.println("\t}");
    }

    /**
     * Create interface with keys for all translations
     * 
     * @param targetDir
     *                Directory of target.
     * @return Name of the interface class
     * @throws IOException
     *                 Writing might fail
     */
    private final String createKeyInterface(final File targetDir)
	    throws IOException {
	final Set<TranslationEntry> data = doc.getTranslations();
	PrintWriter writer = null;
	final String clsName = doc.getClassName() + "Keys";
	try {

	    final File fName = new File(targetDir, clsName + ".java");
	    writer = new PrintWriter(new BufferedWriter(new ASCIIFilterWriter(
		    new OutputStreamWriter(new FileOutputStream(fName),
			    ENCODING))));

	    writer.println("package " + doc.getPackageName() + ";");
	    writer.println();
	    writer.println("/**");
	    writer.println(" * " + doc.getClassComment());
	    writer
		    .println(" * This interface defines all keys to translations.");
	    writer.print(" * Supported doc.getLocales(): ");
	    boolean first = true;
	    for (final LocaleDef l : doc.getLocales()) {
		if (first) {
		    first = false;
		} else {
		    writer.print(", ");
		}
		writer.print(l.getLocale().getDisplayName());
	    }
	    writer.println();

	    writer.println(" */");
	    switch (param.getJdk()) {
	    case JDK14:
		writer.println("public interface " + clsName);
		break;
	    case JDK15:
		writer.println("public enum " + clsName);
		break;
	    }

	    writer.println("{");
	    String currGroup = null;

	    for (final TranslationEntry entry : data) {
		if (!entry.getType().isInResourceBundle()) {
		    continue;
		}
		usedTypes.add(entry.getType());
		final String grp = entry.getGroup();
		if (grp != null && !grp.equals(currGroup)) {
		    // write group header
		    currGroup = grp;
		    writer.println();
		    writer
			    .println("\t//***************************************************************");
		    writer.println("\t//\tGroup: " + currGroup);
		    writer
			    .println("\t//***************************************************************");

		}

		writer.println();
		writer.println("\t/** "
			+ entry.getJavaDoc(doc.getLocales(), "\t *  ", doc));

		writer.println("\t */");
		switch (param.getJdk()) {
		case JDK14:
		    writer.println("\tString " + entry.getKey() + " = \""
			    + entry.getKey() + "\";");
		    break;
		case JDK15:
		    writer.println("\t" + entry.getKey() + ",");
		    break;

		}
	    }
	    writer.println("}");
	    writer.println();

	} finally {
	    if (writer != null) {
		writer.close();
	    }
	}
	return clsName;
    }

    /**
     * Create interface with keys for all translations
     * 
     * @param targetDir
     *                Directory of target.
     * @return Name of the interface class
     * @throws IOException
     *                 Writing might fail
     */
    private final String createEnumInterface(final File targetDir)
	    throws IOException {
	PrintWriter writer = null;
	final Set<TranslationEntry> data = doc.getTranslations();
	final String clsName = doc.getClassName() + "Keys";
	try {

	    final File fName = new File(targetDir, clsName + ".java");
	    writer = new PrintWriter(new BufferedWriter(new ASCIIFilterWriter(
		    new OutputStreamWriter(new FileOutputStream(fName),
			    ENCODING))));

	    writer.println("package " + doc.getPackageName() + ";");
	    writer.println();
	    writer.println("/**");
	    writer.println(" * " + doc.getClassComment());
	    writer
		    .println(" * This interface defines all keys to translations.");
	    writer.print(" * Supported doc.getLocales(): ");
	    boolean first = true;
	    for (final LocaleDef l : doc.getLocales()) {
		if (first) {
		    first = false;
		} else {
		    writer.print(", ");
		}
		writer.print(l.getLocale().getDisplayName());
	    }
	    writer.println();

	    writer.println(" */");
	    writer.println("public interface " + clsName);

	    writer.println("{");
	    String currGroup = null;

	    final List<TranslationEntry> sortByType = new ArrayList<TranslationEntry>(
		    data);
	    Collections.sort(sortByType, SORT_BY_TYPE_COMPARATOR);
	    TranslationDataType currType = null;

	    for (final TranslationEntry entry : sortByType) {
		if (!entry.getType().isInResourceBundle()) {
		    // not all translations are written into resource bundles
		    // e.g. EclipseRCP translations are stored in property
		    // files.
		    continue;
		}

		usedTypes.add(entry.getType());

		// the character behind an enum-value declaration is usually ','
		// but ';' for the last
		// mark as finished, if nothing is to finish
		boolean wasFinished = currType == null;
		if (currType == null || !currType.equals(entry.getType())) {

		    if (currType != null) {
			createEmumMethods(writer, currType);
			wasFinished = true;
			// close last opened inner enumeration
			writer.println("\t}\n");
		    }
		    currType = entry.getType();

		    if (!wasFinished) {
			// the last enum must be finished by ","
			writer.println(",");
			wasFinished = true;
		    }
		    // open next enumeration
		    writer.println("\t/** Enumeration for " + currType + " */");
		    writer.println("\tpublic enum " + currType.getPrefix()
			    + " {");

		    currGroup = null;
		}
		if (!wasFinished) {
		    // the last enum must be finished by ","
		    writer.println(",");
		    wasFinished = true;
		}

		final String grp = entry.getGroup();
		if (grp != null && !grp.equals(currGroup)) {
		    // write group header
		    currGroup = grp;
		    writer.println();
		    writer
			    .println("\t\t//***************************************************************");
		    writer.println("\t\t//\tGroup: " + currGroup);
		    writer
			    .println("\t\t//***************************************************************");

		}

		writer.println();
		writer.println("\t\t/**");
		writer.println("\t\t *  "
			+ entry.getJavaDoc(doc.getLocales(), "\t\t *  ", doc));

		writer.println("\t\t */");
		writer.print("\t\t" + entry.getGroupKey() + "(\""
			+ entry.getKey() + "\")");

	    }
	    if (currType != null) {
		createEmumMethods(writer, currType);
	    }

	    writer.println("\t}");
	    writer.println("}");
	    writer.println();

	} finally {
	    if (writer != null) {
		writer.close();
	    }
	}
	return clsName;
    }

    /**
     * Create helper methods of an key-enumeration
     * 
     * @param writer
     *                Writer code to this writen; null is not allowed
     * @param currType
     *                Current enum type
     */
    private void createEmumMethods(final PrintWriter writer,
	    final TranslationDataType currType) {

	// finish last enum
	writer.println(";");

	// create member for key, constructor and helper methods.
	writer.println();
	writer
		.println("\t\t/** Key of enum object, used in the ResourceBundle as key. */");
	writer.println("\t\tprivate final String key;");
	writer.println();
	writer.println("\t\t/**");
	writer.println("\t\t * Create new enum object with key.");
	writer.println("\t\t * @param key The key string used in the table.");
	writer.println("\t\t */");

	writer.println("\t\tprivate " + currType.getPrefix()
		+ " (final String key) {");
	writer.println("\t\t\tthis.key = key;");
	writer.println("\t\t}");
	writer.println();
	writer.println("\t\t/**");
	writer.println("\t\t * Return key of enum instance.");
	writer.println("\t\t * @return Is never null.");
	writer.println("\t\t */");
	writer.println("\t\tpublic String getKey () {");
	writer.println("\t\t\treturn key;");
	writer.println("\t\t}");
	writer.println();
	writer.println("\t\t/**");
	writer.println("\t\t * Returns the key of enum instance, of its name.");
	writer.println("\t\t * @returnIs never null. ");
	writer.println("\t\t */");
	writer.println("\t\t@Override public String toString () {");
	writer.println("\t\t\treturn getKey();");
	writer.println("\t\t}");
	writer.println();
    }

    /**
     * Create resource file for one language
     * 
     * @param targetDir
     *                Directory of target.
     * @param localeIndex
     *                Index of locale to write
     * @param infName
     *                Name of the created interface
     * @throws IOException
     *                 Writing might fail
     */
    private final void createResourceFile(final File targetDir,
	    int localeIndex, final String infName) throws IOException {

	final Set<TranslationEntry> data = doc.getTranslations();
	PrintWriter writer = null;
	final boolean isDefaultLocale = localeIndex == -1;
	if (isDefaultLocale) {
	    localeIndex = 0;
	}

	// collect imports from entries
	final Set<String> imports = new TreeSet<String>();
	imports.add("java.util.ListResourceBundle");

	for (final TranslationEntry entry : data) {
	    entry.collectImports(imports, localeIndex, doc);
	}

	try {
	    final Locale loc = doc.getLocales().get(localeIndex).getLocale();

	    final String clsName = doc.getClassName()
		    + (isDefaultLocale ? "" : "_" + loc.toString());
	    final File fName = new File(targetDir, clsName + ".java");
	    writer = new PrintWriter(new BufferedWriter(new ASCIIFilterWriter(
		    new OutputStreamWriter(new FileOutputStream(fName),
			    ENCODING))));

	    writer.println("package " + doc.getPackageName() + ";");
	    writer.println();

	    for (final String imp : imports) {
		writer.println("import " + imp + ";");
	    }

	    writer.println("\t/**");
	    writer.println("\t * " + doc.getClassComment());
	    writer.println("\t * Translations for " + loc.getDisplayName());
	    writer.println("\t */");

	    writer.println("public final class " + clsName
		    + " extends ListResourceBundle");
	    writer.println("{");
	    writer.println();

	    writer.println("\t/**");
	    writer.println("\t * Access method to get the translation table.");
	    writer.println("\t * @return The translation table.");
	    writer.println("\t */");
	    switch (param.getJdk()) {
	    case JDK14:

		break;
	    default:
	    case JDK15:
		writer.println("\t@Override");
		break;
	    }
	    writer.println("\tpublic final Object[][] getContents()");

	    writer.println("\t{");
	    writer.println("\t\treturn CONTENTS;");
	    writer.println("\t}");
	    writer.println();
	    writer.println("\t/**");
	    writer.println("\t * The translation table.");
	    writer.println("\t */");
	    writer.println("\tprivate static final Object[][] CONTENTS =");
	    writer.println("\t{");
	    String currGroup = null;

	    for (final TranslationEntry entry : data) {
		if (!entry.getType().isInResourceBundle()) {
		    continue;
		}

		final String grp = entry.getGroup();
		if (grp != null && !grp.equals(currGroup)) {
		    // write group header
		    currGroup = grp;
		    writer.println();
		    writer
			    .println("\t\t//***************************************************************");
		    writer.println("\t\t//\tGroup: " + currGroup);
		    writer
			    .println("\t\t//***************************************************************");

		}

		writer.println();
		writer.println("\t\t/**");

		writer.println("\t\t *  "
			+ entry.getJavaDoc(doc.getLocales(), "\t\t *  ", doc));
		writer.println("\t\t */");

		switch (param.getJdk()) {
		case JDK14:
		    writer.println("\t\t{ " + infName + "." + entry.getKey()
			    + " , " + entry.getValueAsString(localeIndex, doc)
			    + " },");
		    break;
		default:
		case JDK15:
		    writer.println("\t\t{ " + infName + "."
			    + entry.getType().getPrefix() + "."
			    + entry.getGroupKey() + ".getKey() , "
			    + entry.getValueAsString(localeIndex, doc) + " },");
		    break;
		}
	    }
	    writer.println("\t};"); // close CONTENTS
	    writer.println();
	    writer.println("}"); // close class

	} finally {
	    if (writer != null) {
		writer.close();
	    }
	}

    }

    /**
     * Compare translation entries by its type
     */
    private static final Comparator<TranslationEntry> SORT_BY_TYPE_COMPARATOR = new Comparator<TranslationEntry>() {

	/**
	 * Compare by type(first) and group (second)
	 * 
	 * @param o1
	 *                First entry to compare
	 * @param o2
	 *                Second entry to compare
	 * @return See comparable interface for details.
	 */
	@Override
	public int compare(final TranslationEntry o1, final TranslationEntry o2) {
	    int result = o1.getType().compareTo(o2.getType());
	    if (result == 0) {
		result = o1.getGroup().compareTo(o2.getGroup());
		if (result == 0) {
		    result = o1.getKey().compareTo(o2.getKey());
		}
	    }

	    return result;
	}
    };

    /**
     * Do create plugin property files.
     * 
     * @param localeIndex
     *                Index to the file to create.
     * @throws IOException
     *                 Writing might fail
     * 
     * @since 1.0
     */
    private void createPluginProperties(int localeIndex) throws IOException {
	PrintWriter writer = null;

	try {
	    final Locale loc;
	    final String filename;

	    if (localeIndex == -1) {

		// the first is the default.
		localeIndex = 0;
		// loc = doc.getLocales().get(localeIndex);
		filename = "plugin.properties";
	    } else {
		loc = doc.getLocales().get(localeIndex).getLocale();

		// filename = "nl" + File.separator
		// + loc.toString().replace('_', File.separatorChar)
		// + File.separator + "plugin.properties";
		filename = "plugin_" + loc.toString() + ".properties";

	    }

	    final File fName = new File(param.getRcpPluginRoot(), filename);
	    final File pa = fName.getParentFile();
	    if (!pa.exists()) {
		if (!pa.mkdirs()) {
		    throw new IOException("Cant create path: '"
			    + pa.getAbsolutePath() + "'");
		}
	    }

	    String currGroup = null;

	    for (final TranslationEntry entry : doc.getTranslations()) {

		final String grp = entry.getGroup();

		if (entry.getType() != TranslationDataType.EclipseRCP) {

		    // consider only type xml
		    continue;
		}

		if (writer == null) {

		    writer = new PrintWriter(new BufferedWriter(
			    new ASCIIFilterWriter(new OutputStreamWriter(
				    new FileOutputStream(fName), ENCODING))));
		}

		if (grp != null && !grp.equals(currGroup)) {

		    // write group header
		    currGroup = grp;
		    writer.println("");
		    writer
			    .println("#################################################################");
		    writer.println("#\tGroup: " + currGroup);
		    writer
			    .println("#################################################################");
		}

		writer.println("");
		writer.println("#\t" + entry.getPlainKey() + " ("
			+ entry.getType() + ")");
		writer.println(entry.getPlainKey() + "="
			+ entry.getValueAsString(localeIndex, doc));
	    } // end for
	} finally {

	    if (writer != null) {
		writer.close();
	    }
	}
    } // end method createPluginProperties

}
