package de.olafpanz.translationtable;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.TreeMap;

import javax.xml.parsers.ParserConfigurationException;

import org.apache.xmlbeans.XmlException;
import org.apache.xmlbeans.XmlOptions;
import org.xml.sax.SAXException;

import de.olafpanz.xsd.translationTable100.LocaleType;
import de.olafpanz.xsd.translationTable100.TableFileType;
import de.olafpanz.xsd.translationTable100.TranslationTableType;
import de.olafpanz.xsd.translationTable100.TranslationTablesDocument;
import de.olafpanz.xsd.translationTable100.TranslationType;
import de.olafpanz.xsd.translationTable100.ValueType;

/**
 * Implements tool commands.
 * 
 * @author Olaf
 * 
 */
public final class CommandFacade {

    /**
     * Export translation table specified in parameters to given file as xml.
     * 
     * @param param
     *                Startup parameters
     * @param xmlFile
     *                target xml file.
     * @throws IOException
     *                 Error in writing file.
     */
    public static void exportXML(final TranslationTableParameters param,
	    final File xmlFile) throws IOException {

	// create parent path
	final File pa = xmlFile.getParentFile();
	if (!pa.exists()) {
	    if (!pa.mkdirs()) {
		throw new IOException("Creation of path '"
			+ pa.getAbsolutePath() + "' failed.");
	    }
	} else if (!pa.isDirectory()) {
	    throw new IOException("Specified path is a file: '"
		    + pa.getAbsolutePath() + "'");
	}

	// load document
	final List<TranslationDocument> docList = loadDocuments(param);
	final TranslationTablesDocument xmlDoc = toXML(docList);
	docList.clear();

	// write xml:
	final XmlOptions opts = new XmlOptions();
	opts.setCharacterEncoding("US-ASCII");
	opts.setSavePrettyPrint();
	opts.setSavePrettyPrintIndent(2);
	opts.setSavePrettyPrintOffset(2);

	xmlDoc.save(xmlFile, opts);

    }

    /**
     * Convert a translation-document into aa xml-document
     * 
     * @param docList
     *                List of documents to transform. Might be empty but not
     *                null
     * @return Created document
     */
    private static TranslationTablesDocument toXML(
	    final List<TranslationDocument> docList) {
	final TranslationTablesDocument result = TranslationTablesDocument.Factory
		.newInstance();
	final TableFileType xmlFile = result.addNewTranslationTables();

	for (final TranslationDocument doc : docList) {
	    final TranslationTableType ttt = xmlFile.addNewTranslationTable();
	    ttt.setClassName(doc.getClassName());
	    ttt.setComment(doc.getClassComment());
	    ttt.setPackage(doc.getPackageName());
	    ttt.setName(doc.getName());

	    final List<LocaleDef> locales = doc.getLocales();
	    // write locales
	    for (final LocaleDef ld : locales) {
		final LocaleType lt = ttt.addNewLocale();
		lt.setLocale(ld.getLocale().toString());
		if (ld.isDefault()) {
		    lt.setIsMasterLocale(true);
		}
		final Locale dl = ld.getTemplateLocale();
		if (dl != null) {
		    lt.setTemplateLocale(dl.toString());
		}
	    }
	    final int iMax = locales.size();

	    // write translations
	    for (final TranslationEntry te : doc.getTranslations()) {
		final TranslationType tt = ttt.addNewTranslations();
		tt.setKey(te.getPlainKey());
		tt.setGroup(te.getGroup());

		TranslationType.DataType.Enum type;
		switch (te.getType()) {
		case Accelerator:
		    type = TranslationType.DataType.ACCELERATOR;
		    break;
		case EclipseRCP:
		    type = TranslationType.DataType.ECLIPSE_RCP;
		    break;
		case Mnemonic:
		    type = TranslationType.DataType.MNEMONIC;
		    break;
		case Text:
		    type = TranslationType.DataType.TEXT;
		    break;
		case Tooltip:
		    type = TranslationType.DataType.TOOLTIP;
		    break;
		default:
		    throw new IllegalArgumentException("Unknwon type: '"
			    + te.getType() + "'");
		}
		tt.setDataType(type);

		// write texts
		final List<String> trans = te.getTranslations();

		for (int i = 0; i < iMax; i++) {
		    final String tr = trans.get(i);
		    if (tr != null) {
			final ValueType vt = tt.addNewValue();
			vt.setLocale(locales.get(i).getLocale().toString());
			vt.setValue(ASCIIFilterWriter.convert(tr));
		    }
		}

	    }

	}

	// check, if xml is valid
	result.validate();

	return result;
    }

    /**
     * Load all currently specified translation-table-documents
     * 
     * @param param
     *                Current parameters.
     * @return Is never null
     * @throws IOException
     *                 Error in loading
     */
    private static List<TranslationDocument> loadDocuments(
	    final TranslationTableParameters param) throws IOException {
	final File file = param.getFile();
	if (file == null) {
	    throw new IllegalArgumentException(
		    "No file specified for processing.");
	}
	if (!file.exists()) {
	    throw new IllegalArgumentException("File does not exist: '"
		    + file.getAbsolutePath() + "'");
	}

	if (!file.getName().toLowerCase().endsWith("xml")) {
	    try {
		return new ODSLoader(param).load();
	    } catch (final ParserConfigurationException e) {
		throw new IOException("Error in loading file '"
			+ file.getAbsolutePath() + "': " + e.getMessage(), e);
	    } catch (final SAXException e) {
		throw new IOException("Error in loading file '"
			+ file.getAbsolutePath() + "': " + e.getMessage(), e);
	    }

	}
	return loadXml(file);
    }

    /**
     * Load translation table from xml file.
     * 
     * @param file
     *                File must exist and a valid translation-table file
     * @return List of loaded tables
     * @throws IOException
     *                 Error in loading
     */
    private static List<TranslationDocument> loadXml(final File file)
	    throws IOException {
	// load from xml
	try {
	    final TranslationTablesDocument xmlDoc = TranslationTablesDocument.Factory
		    .parse(file);
	    final TableFileType tft = xmlDoc.getTranslationTables();
	    final List<TranslationDocument> result = new ArrayList<TranslationDocument>();

	    for (final TranslationTableType ttt : tft.getTranslationTableList()) {
		final List<LocaleDef> locales = new ArrayList<LocaleDef>();

		// maps from locale identifier to index within locale array
		final Map<String, Integer> localeMap = new TreeMap<String, Integer>();

		int index = 0;
		for (final LocaleType lt : ttt.getLocaleList()) {
		    final LocaleDef def = new LocaleDef(ODSLoader.toLocale(lt
			    .getLocale()), ODSLoader.toLocale(lt
			    .getTemplateLocale()), lt.getIsMasterLocale());
		    locales.add(def);
		    localeMap.put(lt.getLocale(), index++);
		}

		final TranslationDocument doc = new TranslationDocument(ttt
			.getClassName(), ttt.getPackage(), ttt.getComment(),
			ttt.getName(), locales);

		result.add(doc);

		for (final TranslationType ty : ttt.getTranslationsList()) {

		    final TranslationDataType type;
		    final TranslationType.DataType.Enum dt = ty.getDataType();
		    if (TranslationType.DataType.ACCELERATOR.equals(dt)) {
			type = TranslationDataType.Accelerator;
		    } else if (TranslationType.DataType.ECLIPSE_RCP.equals(dt)) {
			type = TranslationDataType.EclipseRCP;
		    } else if (TranslationType.DataType.MNEMONIC.equals(dt)) {
			type = TranslationDataType.Mnemonic;
		    } else if (TranslationType.DataType.TEXT.equals(dt)) {
			type = TranslationDataType.Text;
		    } else if (TranslationType.DataType.TOOLTIP.equals(dt)) {
			type = TranslationDataType.Tooltip;
		    } else {
			throw new IllegalArgumentException("Unhandled type: '"
				+ dt + "'");
		    }

		    final String trans[] = new String[localeMap.size()];
		    // final List<String> translations = new ArrayList<String>(
		    // localeMap.size());

		    for (final ValueType vt : ty.getValueList()) {
			final Integer idx = localeMap.get(vt.getLocale());
			if (idx == null) {
			    throw new IOException(
				    "Locale '"
					    + vt.getLocale()
					    + "' not specified in locale section of file.");
			}

			trans[idx] = vt.getValue();
		    }
		    final TranslationEntry entry = new TranslationEntry(ty
			    .getKey(), ty.getGroup(), type,
			    ty.getDescription(), Arrays.asList(trans));
		    doc.addTranslation(entry);
		}

	    }
	    return result;

	} catch (final XmlException e) {
	    throw new IOException("Error in loading file '"
		    + file.getAbsolutePath() + "': " + e.getMessage(), e);
	}

    }

    /**
     * Create translations tables for one file
     * 
     * @param param
     *                Parameters to create table.
     * @throws IOException
     *                 File loading might mail.
     */
    public static void createCode(final TranslationTableParameters param)
	    throws IOException {

	final List<TranslationDocument> docList = loadDocuments(param);

	for (final TranslationDocument doc : docList) {
	    final CodeGenerator gen = new CodeGenerator(doc, param);
	    gen.createFiles();
	}

    }
}
