/**
 * Copyright 2005-2008 Olaf Panz 
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 *
 * 		http://www.apache.org/licenses/LICENSE-2.0 
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 */
package de.olafpanz.translationtable;

import java.io.File;

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Task;

/**
 * Privde I18N code generator as ant task.
 * 
 * @author Olaf Panz
 */
public final class I18NTask extends Task {

    /**
     * Parameters to create translation tables.
     */
    private final TranslationTableParameters params = new TranslationTableParameters();

    /**
     * Do execute the ant task.
     * 
     * @throws BuildException
     *                 Is thrown in case of an error.
     */
    @Override
    public void execute() throws BuildException {

	try {
	    CommandFacade.createCode(params);

	} catch (final Throwable e) {
	    e.printStackTrace();
	    throw new BuildException("I18N failed", e);
	}
    } // end method execute

    /**
     * Set the file that should be executed.
     * 
     * @param file
     *                The file to process.
     * 
     * @since 1.0
     */
    public void setODSFile(final String file) {
	params.setOds(new File(file));
    }

    /**
     * Set jdk to create source code for,
     * 
     * @param jdkName
     *                Name of jdk. Use jdk 1.4, if string contains a 4
     */
    public void setJDK(final String jdkName) {
	final JDKDefine jdk;
	if ("1.4".equals(jdkName) || "4".equals(jdkName)) {
	    jdk = JDKDefine.JDK14;
	} else if ("1.5".equals(jdkName) || "5".equals(jdkName)) {
	    jdk = JDKDefine.JDK15;
	} else if ("1.6".equals(jdkName) || "6".equals(jdkName)) {
	    jdk = JDKDefine.JDK15;
	} else {
	    throw new BuildException("Unknown JDK: '" + jdkName + "'");
	}

	params.setJdk(jdk);
    }

    /**
     * Set the environment (client/server).
     * 
     * @param envStr
     *                Null is not allowed
     */
    public void setEnvironment(final String envStr) {
	final Environment env;
	if (envStr.equalsIgnoreCase("server")) {
	    env = Environment.Server;
	} else if (envStr.equalsIgnoreCase("client")) {
	    env = Environment.Client;
	} else {
	    throw new BuildException("Unknown environment: '" + envStr + "'");
	}
	params.setEnvironment(env);

    }

    /**
     * Set root of rcp plugin.
     * 
     * @param root
     *                Null is not allowed.
     */
    public void setRcpPluginRoot(final String root) {
	params.setRcpPluginRoot(new File(root));

    }

    /**
     * Set root path. That is the path where the source is created to.
     * 
     * @param root
     *                Null is not allowed.
     */
    public void setSourceRoot(final String root) {
	params.setSourceRoot(new File(root));
    }

} // end class I18NTask
