/**
 * Copyright 2005-2008 Olaf Panz 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 *
 * 		http://www.apache.org/licenses/LICENSE-2.0 
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 */
package de.olafpanz.translationtable;

import java.io.BufferedInputStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

/**
 * Create translation files from .odt table: - Interface that defines all
 * translation keys. This ensures that keys are only used once - Translation
 * table for all specified languages. - Error messages for wrong formats
 * 
 * @author Olaf Panz
 */
final class ODSLoader {

    /**
     * Name of the content file within the ods file.
     */
    private static final String CONTENT_NAME = "content.xml";

    /**
     * Create translations tables for one file
     * 
     * @return List of loaded translation tables.
     * @throws IOException
     *                 File loading might mail.
     * @throws ParserConfigurationException
     *                 Parsing of document might fail
     * @throws SAXException
     *                 Parsing of document might fail
     */
    public List<TranslationDocument> load() throws IOException,
	    ParserConfigurationException, SAXException {
	List<TranslationDocument> result = null;
	final String err = param.isComplete();
	if (err != null) {
	    throw new IllegalArgumentException("Parameters are incomplete: "
		    + err);
	}

	// we access a file within a zip:
	ZipInputStream in = null;

	try {
	    in = new ZipInputStream(new BufferedInputStream(
		    new FileInputStream(param.getFile())));
	    System.out.println("Processing (" + param.getJdk() + ","
		    + param.getEnvironment() + "): "
		    + param.getFile().getAbsolutePath());
	    do {
		final ZipEntry entry = in.getNextEntry();

		if (entry == null) {
		    break;
		}
		if (CONTENT_NAME.equals(entry.getName())) {
		    // this is the file we need!
		    final DocumentBuilderFactory builderFab = DocumentBuilderFactory
			    .newInstance();
		    builderFab.setIgnoringComments(true);
		    builderFab.setNamespaceAware(true);
		    final DocumentBuilder builder = builderFab
			    .newDocumentBuilder();
		    doc = builder.parse(in);

		    // final ODSLoader process = new ODSLoader(doc, param);
		    result = readSheet();

		    break;// we are ready now

		}
		in.closeEntry();
	    } while (true);
	} finally {
	    if (in != null) {
		in.close();
	    }
	}

	return result;
    }

    /**
     * The document to process.
     */
    private Document doc;

    /**
     * Parameters for creation.
     */
    private final TranslationTableParameters param;

    /**
     * Create new class to create the translation table.
     * 
     * @param param
     *                Parameters for code generation
     */
    public ODSLoader(final TranslationTableParameters param) {
	if (param == null) {
	    throw new IllegalArgumentException("param is null.");
	}
	this.param = param;
    }

    /**
     * Create locale from string
     * 
     * @param locString
     *                The locale string
     * @return The created locale.
     */
    static final Locale toLocale(final String locString) {
	if (locString == null) {
	    return null;
	}

	final String[] elements = locString.split("_");
	final Locale result;

	switch (elements.length) {
	case 1:
	    result = new Locale(elements[0]);
	    break;
	case 2:
	    result = new Locale(elements[0], elements[1]);
	    break;
	case 3:
	    result = new Locale(elements[0], elements[1], elements[2]);
	    break;

	default:
	    result = null;
	    break;

	}

	return result;
    }

    /**
     * Read a column from a given row.
     * 
     * @param rows
     *                List of rows in xml
     * @param row
     *                The row to use
     * @param column
     *                The column index
     * @return null, if value is not specified.
     */
    private final String readCellValue(final NodeList rows, final int row,
	    final int column) {
	if (row >= rows.getLength()) {
	    // row does not exists
	    return null;
	}
	final NodeList cells = ((Element) rows.item(row))
		.getElementsByTagName("table:table-cell");

	// read until cell
	int cellIdx = 0;
	int xmlIdx = 0;
	final int xmlIdxMax = cells.getLength();

	while (xmlIdx < xmlIdxMax) {
	    final Element cell = (Element) cells.item(xmlIdx);

	    // check repeat counter:
	    final String repeat = cell
		    .getAttribute("table:number-columns-repeated");
	    if (repeat != null && repeat.length() > 0) {
		cellIdx += Integer.parseInt(repeat);
	    } else {
		cellIdx++;
	    }

	    if (column < cellIdx) {
		// found!
		break;
	    }

	    xmlIdx++;
	}

	if (xmlIdx >= cells.getLength()) {
	    // not found
	    return null;
	}

	final NodeList textpList = ((Element) cells.item(xmlIdx))
		.getElementsByTagName("text:p");
	if (textpList.getLength() == 0) {
	    // no text!!??
	    return null;
	}

	final Element textp = (Element) textpList.item(0);
	final String result = textp.getTextContent();

	return result;
    }

    /**
     * Read the all pages from document.
     * 
     * @return List of loaded documents.
     * 
     */
    private final List<TranslationDocument> readSheet() {
	final Element documentRoot = doc.getDocumentElement();
	final Element body = (Element) documentRoot.getElementsByTagName(
		"office:body").item(0);
	final Element spreadsheet = (Element) body.getElementsByTagName(
		"office:spreadsheet").item(0);
	final NodeList tables = spreadsheet.getElementsByTagName("table:table");

	final List<TranslationDocument> result = new ArrayList<TranslationDocument>();
	// read table by table
	final int iMax = tables.getLength();
	for (int i = 0; i < iMax; i++) {
	    final TranslationDocument table = readTable((Element) tables
		    .item(i));
	    result.add(table);

	}
	return result;
    }

    /**
     * Process one table
     * 
     * @param table
     *                The table to process.
     * @return Table read from file.
     */
    private final TranslationDocument readTable(final Element table) {
	final String tableName = table.getAttribute("table:name");
	System.out.println("    " + tableName);

	final NodeList rows = table.getElementsByTagName("table:table-row");
	if (rows.getLength() < 6) {
	    // table contains no data!
	    System.err.println(tableName
		    + " does not contain translation data.");
	    return null;
	}

	final String packageName = readCellValue(rows, 1, 1);
	if (packageName == null) {
	    System.err.println(tableName
		    + ": Package name not specified! (Cell B2)");
	    return null;
	}

	final String name = readCellValue(rows, 2, 1);
	if (name == null) {
	    System.err.println(tableName + ": Name not specified! (Cell B3)");
	    return null;
	}

	final String classComment = readCellValue(rows, 3, 1);
	if (classComment == null) {
	    System.err.println(tableName
		    + ":  Class-comment is missing! (Cell B4)");
	    return null;
	}

	// Read all locales
	final List<Locale> locales = new ArrayList<Locale>();
	do {
	    final String locString = readCellValue(rows, 5, 4 + locales.size());
	    if (locString == null) {
		// no more locales!
		break;
	    }
	    final Locale loc = toLocale(locString);
	    if (loc == null) {
		System.err.println(tableName + ": Error in Locale format: \""
			+ locString + "\"");
		return null;
	    }

	    locales.add(loc);

	} while (true);

	final List<LocaleDef> localeDefs = new ArrayList<LocaleDef>();
	final int locCount = locales.size();

	for (int i = 0; i < locCount; i++) {
	    final String locString = readCellValue(rows, 6, 4 + i);
	    final Locale defLoc = locString == null ? null
		    : toLocale(locString);
	    localeDefs.add(new LocaleDef(locales.get(i), defLoc, i == 0));

	}

	final TranslationDocument tTable = new TranslationDocument(name,
		packageName, classComment, tableName, localeDefs);

	// read data rows
	int idx = 7;

	do {
	    final String plainKey = readCellValue(rows, idx, 0);
	    if (plainKey == null) {
		// we are ready!
		break;
	    }

	    final String group = readCellValue(rows, idx, 1);

	    final String typeID = readCellValue(rows, idx, 2);
	    if (typeID == null) {
		System.err.println(tableName + ": No type defined for: \""
			+ plainKey + "\" row is ignored!");
		idx++;
		continue;
	    }

	    // evaluate enum for string:
	    final TranslationDataType type = TranslationDataType
		    .valueOf(typeID);

	    if (type == null) {
		System.err.println(tableName + ": Type undefined: \"" + typeID
			+ "\" for: \"" + plainKey + "\" row is ignored!");
		idx++;
		continue;
	    }

	    final String fullKey = TranslationEntry.createKey(type, group,
		    plainKey);

	    final String desc = readCellValue(rows, idx, 3);

	    final List<String> translations = new ArrayList<String>();

	    // read all translations
	    for (int i = 0; i < locCount; i++) {
		String tr = readCellValue(rows, idx, 4 + i);
		if (tr == null || tr.length() == 0) {
		    tr = null;
		}
		translations.add(tr);
	    }

	    // now we have read all columns, try now to resolve all missing
	    // values.

	    final TranslationEntry row = new TranslationEntry(plainKey, group,
		    type, desc, translations);

	    if (!tTable.addTranslation(row)) {
		System.err.println(tableName + ": double key in table :\""
			+ fullKey + "\" The last will win!");
	    }

	    idx++;
	} while (true);

	return tTable;
    }

    // /**
    // * Resolve a value for a given column from the template language(s).
    // *
    // * @param translations
    // * List of all translations
    // * @param templateLocales
    // * List of template locales (indeces to columns)
    // * @param i
    // * Current column to consider.
    // * @param way
    // * Because the method is considering templates is several
    // * level, we check here circular dependencies.
    // * @return The resolved string, might be null.
    // */
    // private final String resolveTemplateLanguage(
    // final List<String> translations,
    // final List<Integer> templateLocales, final int i,
    // final Set<Integer> way) {
    // final Integer tempLocale = templateLocales.get(i);
    // if (tempLocale == null) {
    // // no template language!
    // return null;
    // }
    // if (way.contains(tempLocale)) {
    // // circular!
    // System.err.println("Circular dependency!");
    // return null;
    // }
    // way.add(i);
    //
    // // get value:
    // final String value = translations.get(tempLocale);
    // if (value != null && value.length() > 0) {
    // return value;
    // }
    // // also empty, check for next step dependencies
    // return resolveTemplateLanguage(translations, templateLocales,
    // tempLocale, way);
    //
    // }

}
