package de.olafpanz.translationtable;

import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Set;
import java.util.TreeSet;

/**
 * Contains a loaded translation document.
 * 
 * @author Olaf Panz
 * 
 */
final class TranslationDocument {

    /**
     * All translation entries.
     */
    private final Set<TranslationEntry> translations = new TreeSet<TranslationEntry>();

    /**
     * List of supported locales in order or usage in data.
     */
    private final List<LocaleDef> locales;
    /**
     * Base name of table within multi-table file.
     */
    private final String name;
    /**
     * Name of package.
     */
    private final String packageName;
    /**
     * Class base name.
     */
    private final String className;
    /**
     * Comment for generated class.
     */
    private final String classComment;

    /**
     * Search for an locale within a list of LocaleDef
     * 
     * @param lst
     *                Null is not allowed.
     * @param loc
     *                Null is not allowed.
     * @return Is -1, if locale was not found.
     */
    private static int getIndex(final List<LocaleDef> lst, final Locale loc) {
	for (int i = 0; i < lst.size(); i++) {
	    if (lst.get(i).getLocale().equals(loc)) {
		return i;
	    }
	}
	return -1;
    }

    /**
     * Create new document
     * 
     * @param className
     *                Class base name.
     * @param packageName
     *                Name of package.
     * @param classComment
     *                Comment for generated class.
     * @param name
     *                Base name of table within multi-table file.
     * @param locales
     *                List of locales in the document
     */
    public TranslationDocument(final String className,
	    final String packageName, final String classComment,
	    final String name, final List<LocaleDef> locales) {

	this.classComment = classComment;
	this.className = className;
	this.locales = locales;

	this.name = name;
	this.packageName = packageName;

	// create default locale array
	templateLocales = new int[locales.size()];
	for (int i = 0; i < templateLocales.length; i++) {
	    final LocaleDef def = locales.get(i);
	    final Locale defaultLoc = def.getTemplateLocale();
	    templateLocales[i] = -1;
	    if (defaultLoc != null) {
		templateLocales[i] = getIndex(locales, defaultLoc);
		// check, if an index was found
		if (templateLocales[i] == -1) {
		    throw new IllegalArgumentException("Default locale '"
			    + defaultLoc + "' for locale '" + locales.get(i)
			    + "' not found.");
		}
	    }
	}

	// check for recursions
	for (int i = 0; i < templateLocales.length; i++) {
	    final Set<Integer> way = new TreeSet<Integer>();
	    int idx = i;
	    do {
		// check, if new position was already visited
		if (way.contains(idx)) {

		    throw new IllegalArgumentException(
			    "Circular default locale dependencies for '"
				    + locales.get(i));
		}
		way.add(idx);

		idx = templateLocales[idx];

	    } while (idx >= 0);

	}

	// check for double master locales
	boolean hasMaster = false;
	for (final LocaleDef def : locales) {

	    if (def.isDefault()) {
		if (hasMaster) {
		    throw new IllegalArgumentException(
			    "Table has multiple master-locales, only one is allowed.");
		}
		hasMaster = true;
	    }

	}

    }

    /**
     * All translation entries.
     * 
     * @return Set is unmodifiable
     */
    public Set<TranslationEntry> getTranslations() {
	return Collections.unmodifiableSet(translations);
    }

    /**
     * Returns index of template locale
     * 
     * @param localeIndex
     *                Index of locale for that the default locale is requested
     * @return The index, is -1, if no default locale exists.
     */
    public int getTemplateLocaleIndex(final int localeIndex) {
	return templateLocales[localeIndex];
    }

    /**
     * The default locale is the locale that is used if no specific translations
     * exists for a locale
     * 
     * @return Index of locale
     */
    public int getDefaultLocale() {
	final int iMax = locales.size();
	for (int i = 0; i < iMax; i++) {
	    if (locales.get(i).isDefault()) {
		return i;
	    }
	}
	return 0; // 0 is the default default-locale
    }

    /**
     * Default locales of locale, positions correspond to {@link #locales}.
     */
    private final int[] templateLocales;

    /**
     * List of supported locales in order or usage in data.
     * 
     * @return List is unmodifiable
     */
    public List<LocaleDef> getLocales() {
	return Collections.unmodifiableList(locales);
    }

    /**
     * @return Base name of table within multi-table file.
     */
    public String getName() {
	return name;
    }

    /**
     * @return Name of package.
     */
    public String getPackageName() {
	return packageName;
    }

    /**
     * @return Comment for generated class.
     */
    public String getClassName() {
	return className;
    }

    /**
     * @return Class base name.
     */
    public String getClassComment() {
	return classComment;
    }

    /**
     * Add a translation key
     * 
     * @param entry
     *                null is not allowed.
     * @return true, if entry was added, false, if entry with same key allready
     *         exists.
     */
    public boolean addTranslation(final TranslationEntry entry) {

	return translations.add(entry);
    }

    /**
     * Check, if at least one Bundle-Translation exists
     * 
     * @return true, if yes
     */
    public boolean hasBundle() {
	for (final TranslationEntry entry : translations) {
	    switch (entry.getType()) {
	    case Accelerator:
		return true;
	    case Mnemonic:
		return true;
	    case Text:
		return true;
	    case Tooltip:
		return true;
	    case EclipseRCP:
		break;
	    }
	}
	return false;

    }

    /**
     * Check, if at least one RCP-Translation exists
     * 
     * @return true, if yes
     */
    public boolean hasRCP() {
	for (final TranslationEntry entry : translations) {
	    switch (entry.getType()) {
	    case Accelerator:
		break;
	    case Mnemonic:
		break;
	    case Text:
		break;
	    case Tooltip:
		break;
	    case EclipseRCP:
		return true;
	    }
	}
	return false;
    }

}
