/**
 * Copyright 2005-2008 Olaf Panz 
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 *
 * 		http://www.apache.org/licenses/LICENSE-2.0 
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 */

package de.olafpanz.translationtable;

import java.util.List;
import java.util.Set;

/**
 * Contains one translation entry.
 * 
 * @author Olaf Panz
 * 
 */
final class TranslationEntry implements Comparable<TranslationEntry> {
    /**
     * Create a key from type and plan key.
     * 
     * @param type
     *                The type
     * @param plainKey
     *                The plain key.
     * @param group
     *                Group of this translation value.
     * @return The created full key.
     */
    public static String createKey(final TranslationDataType type,
	    final String group, final String plainKey) {

	final StringBuilder buf = new StringBuilder();
	buf.append(type.getPrefix()).append('_');
	if (group != null) {
	    buf.append(group.replace('.', '_')).append('_');
	}
	buf.append(plainKey.replace('.', '_'));

	return buf.toString();

    }

    /**
     * Debug only
     * 
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
	return getKey();
    }

    /**
     * Key to this entry
     */
    private final String key;

    /**
     * Type of this entry
     */
    private final TranslationDataType type;

    /**
     * Description of this entry
     */
    private final String desc;

    /**
     * List of translation, is corresponding to locales in TranslationTable.
     */
    private final List<String> translations;

    /**
     * A translation entry might belong to a group. This might be null.
     */
    private final String group;

    /**
     * Create a translation entry
     * 
     * @param key
     *                Key to this entry
     * @param group
     *                A translation entry might belong to a group. This might be
     *                null.
     * @param type
     *                Type of this entry
     * @param desc
     *                Desription of this entry
     * @param translations
     *                List of translation, is corresponding to locales in
     *                TranslationTable.
     */
    public TranslationEntry(final String key, final String group,
	    final TranslationDataType type, final String desc,
	    final List<String> translations) {
	assert key != null;
	assert type != null;
	assert desc != null;
	assert translations != null;

	this.key = key;
	this.group = group;
	this.type = type;
	this.desc = desc;
	this.translations = translations;
    }

    /**
     * Compare with another translation entry by key.
     * 
     * @param arg0
     *                The other entry
     * @return see declaration
     */
    public int compareTo(final TranslationEntry arg0) {
	// first sorting criteria is group, second name, third type:
	final String g1 = getGroup();
	final String g2 = arg0.getGroup();
	int result = 0;
	if (g1 == null && g2 != null) {
	    result = -1;
	} else if (g1 != null && g2 == null) {
	    result = 1;
	}
	if (result == 0 && g1 != null) {
	    result = getGroup().compareTo(arg0.getGroup());
	}

	if (result == 0) {
	    result = getPlainKey().compareTo(arg0.getPlainKey());

	    if (result == 0) {
		result = getType().compareTo(arg0.getType());
	    }
	}

	return result;
    }

    /**
     * Convert a declarative string to code that creates a character.
     * 
     * @param str
     *                The declarative String
     * @return The code fragment
     */
    private final String convert2Character(final String str) {

	final StringBuilder buf = new StringBuilder();

	buf.append("new Character ('");
	final boolean isUnicode = str.startsWith("\\u") && str.length() == 6;

	if (str.length() != 1 && !isUnicode) {
	    buf.append('?');
	    System.err
		    .println("Only one character is allowed for mnemonics, buf found '"
			    + str + "' for key '" + getGroupKey() + "'");
	} else {
	    buf.append(str);
	}
	buf.append("')");

	return buf.toString();

    }

    /**
     * Convert a descriptive string to a key stroke define.
     * 
     * Format: Allowed is a combination of one key expression and a list of
     * modifiers. All keys defined in KeyEvent are allowed. VK_ can be written,
     * but it is added, if not present. It is also possible to specify a
     * character value by \\uXXXX Allowed modifiers: SHIFT, CTRL, META, ALT. The
     * elements are combined by "+". Examples: "Alt+F4", "E+SHIFT",
     * "A+Shift+Alt", "VK_E","\\u4711+SHIFT"
     * 
     * @param str
     *                The descriptive string
     * @return The generated code fragment
     */
    private final String convert2KeyStroke(final String str) {
	final StringBuilder buf = new StringBuilder();

	boolean isShift = false;
	boolean isCtrl = false;
	boolean isMeta = false;
	boolean isAlt = false;
	boolean isCharacter = false;
	String keyStroke = null;

	final String[] elements = str.toUpperCase().split("\\+");
	for (final String element : elements) {
	    final String elem = element.trim();

	    if ("SHIFT".equals(elem)) {
		isShift = true;
	    } else if ("CTRL".equals(elem)) {
		isCtrl = true;
	    } else if ("META".equals(elem)) {
		isMeta = true;
	    } else if ("ALT".equals(elem)) {
		isAlt = true;
	    } else if (elem.startsWith("\\U")) {
		isCharacter = true;
		keyStroke = elem.toLowerCase();
	    } else {
		isCharacter = false;
		if (elem.startsWith("VK_")) {
		    keyStroke = elem;
		} else {
		    keyStroke = "VK_" + elem;
		}
	    }
	}

	if (!isCharacter && keyStroke == null) {
	    System.err.println(keyStroke
		    + ": Invalid key stroke definition for accelerator: \""
		    + str + "\"");
	    return "null";
	}

	// generate string:
	buf.append("KeyStroke.getKeyStroke (");
	if (isCharacter) {
	    buf.append("new Character ('");
	    buf.append(keyStroke);
	    buf.append("')");
	} else {
	    buf.append("KeyEvent.");
	    buf.append(keyStroke);
	}
	buf.append(",");
	boolean orNeeded = false;
	if (isShift) {
	    buf.append("InputEvent.SHIFT_MASK");
	    orNeeded = true;
	}
	if (isCtrl) {
	    if (orNeeded) {
		buf.append(" | ");
	    }
	    buf.append("InputEvent.CTRL_MASK");
	    orNeeded = true;
	}
	if (isMeta) {
	    if (orNeeded) {
		buf.append(" | ");
	    }
	    buf.append("InputEvent.META_MASK");
	    orNeeded = true;
	}
	if (isAlt) {
	    if (orNeeded) {
		buf.append(" | ");
	    }
	    buf.append("InputEvent.ALT_MASK");
	    orNeeded = true;
	}
	buf.append(")");

	return buf.toString();
    }

    /**
     * 
     * @return Description of this entry
     */
    public String getDesc() {
	return desc;
    }

    /**
     * Create java doc comment of this entry.
     * 
     * @param locales
     *                Current locale list, used to create list of translations
     * @param indend
     *                Indend, used for new line.
     * @param doc
     *                Document is needed to resolve translation string.
     * @return Created string
     */
    public String getJavaDoc(final List<LocaleDef> locales,
	    final String indend, final TranslationDocument doc) {
	final StringBuilder buf = new StringBuilder();
	buf.append(getDesc()).append(" (").append(getType()).append(")");

	buf.append("\n").append(indend).append("Translations:");
	for (int i = 0; i < locales.size(); i++) {
	    buf.append("\n").append(indend).append("   ").append(
		    locales.get(i).getLocale().getDisplayName()).append(": \"")
		    .append(getRawTranslationString(i, doc)).append("\"");
	}

	return buf.toString();
    }

    /**
     * 
     * @return A translation entry might belong to a group. This might be null.
     */
    public String getGroup() {
	return group;
    }

    /**
     * 
     * @return The key with type prefix.
     */
    public String getKey() {
	return createKey(type, group, getPlainKey());
    }

    /**
     * Return combination of group and key
     * 
     * @return Is never null
     */
    public String getGroupKey() {
	final StringBuilder buf = new StringBuilder();

	if (group != null) {
	    buf.append(group.replace('.', '_')).append('_');
	}
	buf.append(key.replace('.', '_'));

	return buf.toString();
    }

    /**
     * This is the key without type prefix
     * 
     * @return Key to this entry
     */
    public String getPlainKey() {
	return key;
    }

    /**
     * 
     * @return List of translation, is corresponding to locales in
     *         TranslationTable.
     */
    public List<String> getTranslations() {
	return translations;
    }

    /**
     * 
     * @return Type of this entry
     */
    public TranslationDataType getType() {
	return type;
    }

    /**
     * Request the raw translation string for a language at an index position.
     * Default values are resolved here.
     * 
     * Note that the document will fix the data model for endless recursions.
     * 
     * @param translationIndex
     *                Index of requested translation.
     * @param doc
     *                Document is needed for default language resolving.
     * @return Is never null.
     */
    private String getRawTranslationString(final int translationIndex,
	    final TranslationDocument doc) {
	String result = null;
	int idx = translationIndex;
	do {
	    result = translations.get(idx);
	    if (result != null) {
		break;
	    }
	    idx = doc.getTemplateLocaleIndex(idx);
	} while (idx >= 0);
	return result;
    }

    /**
     * Evaluate the string that should be written to file for a given index.
     * 
     * @param translationIndex
     *                The index of the tanslation
     * @param doc
     *                Document is needed to resolve derived languages.
     * @return The Source code fragment.
     */
    public String getValueAsString(final int translationIndex,
	    final TranslationDocument doc) {
	final String str = getRawTranslationString(translationIndex, doc);
	final String result;

	if (str == null) {
	    result = "null";
	} else {
	    switch (type) {
	    case Text:
	    case Tooltip:
		result = "\"" + str + "\"";
		break;
	    case Accelerator:
		result = convert2KeyStroke(str);
		break;

	    case Mnemonic:
		result = convert2Character(str);
		break;
	    case EclipseRCP:
		result = str;
		break;

	    default:
		assert false : "Unhandled type!: " + type;
		result = "null";
		break;
	    }
	}
	return result;
    }

    /**
     * Collect all used imports
     * 
     * @param imports
     *                Fill this set with results.
     * @param localeIndex
     *                Index of locale to collect imports for.
     * @param doc
     *                Document is needed to resolve derived languages.
     */
    public void collectImports(final Set<String> imports,
	    final int localeIndex, final TranslationDocument doc) {

	final String str = getValueAsString(localeIndex, doc);
	if (str.contains("InputEvent")) {
	    imports.add("java.awt.event.InputEvent");
	}
	if (str.contains("KeyStroke")) {
	    imports.add("javax.swing.KeyStroke");
	}
	if (str.contains("KeyEvent")) {
	    imports.add("java.awt.event.KeyEvent");
	}

    }
}
