/**
 * Copyright 2005-2008 Olaf Panz 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 *
 * 		http://www.apache.org/licenses/LICENSE-2.0 
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 */
package de.olafpanz.translationtable;

import java.io.File;
import java.io.IOException;

/**
 * Create translation files from .odt table: - Interface that defines all
 * translation keys. This ensures that keys are only used once - Translation
 * table for all specified languages. - Error messages for wrong formats
 * <p>
 * Command line parameters for testing:
 * <p>
 * -rs src\test\java -jdk 4 -client -ods TestJDK1.4-client.ods -codegen -rxe
 * target/TestJDK1.4-client.xml -server -ods TestJDK1.4-server.ods -codegen -rxe
 * target/TestJDK1.4-server.xml -jdk 5 -client -ods TestJDK1.5-client.ods -rxe
 * target/TestJDK1.5-client.xml -codegen -server -ods TestJDK1.5-server.ods
 * -codegen -rxe target/TestJDK1.5-server.xml -ods Empty.ods -codegen -rxe
 * target/Empty.xml -xml target/TestJDK1.4-client.xml -client -jdk 4 -codegen
 * -xml target/TestJDK1.4-server.xml -server -jdk 4 -codegen -xml
 * target/TestJDK1.5-client.xml -client -jdk 5 -codegen -xml
 * target/TestJDK1.5-server.xml -server -jdk 5 -codegen -h
 * 
 * @author Olaf Panz
 */
final class TranslationTableMain {

    /**
     * Specify the sheet document to (.ods) as first parameter
     * 
     * @param args
     *                Command-line arguments.
     * @throws IOException
     *                 File loading might mail.
     * 
     */
    public static void main(final String[] args) throws IOException {
	System.out.println("TranslationTable (c) 2005-2008 by Olaf Panz");

	final TranslationTableParameters param = new TranslationTableParameters();

	final String usrDir = System.getProperty("user.dir");
	ArgumentState state = ArgumentState.Default;

	if (args.length == 0) {
	    printHelp();
	}

	for (final String arg : args) {
	    final ArgumentState temp = state;
	    state = ArgumentState.Default;

	    switch (temp) {
	    case Default:
		if ("-h".equals(arg)) {
		    printHelp();
		} else if ("-ods".equals(arg)) {
		    state = ArgumentState.ReadODS;
		} else if ("-xml".equals(arg)) {
		    state = ArgumentState.ReadXML;
		} else if ("-as".equals(arg)) {
		    state = ArgumentState.AbsoluteSourcePath;
		} else if ("-rs".equals(arg)) {
		    state = ArgumentState.RelativeSourcePath;
		} else if ("-ae".equals(arg)) {
		    state = ArgumentState.AbsoluteRCPPath;
		} else if ("-re".equals(arg)) {
		    state = ArgumentState.RelativeRCPPath;
		} else if ("-axe".equals(arg)) {
		    state = ArgumentState.AbsoluteXMLExport;
		} else if ("-rxe".equals(arg)) {
		    state = ArgumentState.RelativeXMLExport;
		} else if ("-jdk".equals(arg)) {
		    state = ArgumentState.JDK;
		} else if ("-server".equals(arg)) {
		    param.setEnvironment(Environment.Server);

		} else if ("-client".equals(arg)) {
		    param.setEnvironment(Environment.Client);
		} else if ("-codegen".equals(arg)) {
		    CommandFacade.createCode(param);

		} else {
		    printHelp();
		    throw new IllegalArgumentException("Unknown parameter: "
			    + arg);
		}
		break;
	    case JDK:
		if (arg.indexOf('4') != -1) {
		    param.setJdk(JDKDefine.JDK14);
		} else {
		    param.setJdk(JDKDefine.JDK15);
		}

		break;
	    case AbsoluteSourcePath:
		param.setSourceRoot(new File(arg));
		break;
	    case RelativeSourcePath:
		param.setSourceRoot(new File(usrDir, arg));
		break;
	    case AbsoluteRCPPath:
		param.setRcpPluginRoot(new File(arg));
		break;
	    case RelativeRCPPath:
		param.setRcpPluginRoot(new File(usrDir, arg));
		break;
	    case ReadODS:
		param.setOds(new File(usrDir, arg));
		break;
	    case ReadXML:
		param.setXml(new File(usrDir, arg));
		break;
	    case AbsoluteXMLExport:
		CommandFacade.exportXML(param, new File(arg));
		break;
	    case RelativeXMLExport:
		CommandFacade.exportXML(param, new File(usrDir, arg));
		break;
	    }
	}

    }

    /**
     * States when arguments are read
     * 
     * @author Olaf Panz
     * 
     */
    private enum ArgumentState {
	/**
	 * Next is ODS or parameter prefix
	 */
	Default,

	/**
	 * Export last loaded translation table to xml with absolute path
	 * specification
	 */
	AbsoluteXMLExport,

	/**
	 * Export last loaded translation table to xml with relative path
	 * specification
	 */
	RelativeXMLExport,

	/**
	 * Next is JDK number
	 */
	JDK,

	/**
	 * Next is an absoulte source path
	 */
	AbsoluteSourcePath,

	/**
	 * Next is a relative source path.
	 */
	RelativeSourcePath,

	/**
	 * Next argument is absolute rcp path.
	 */
	AbsoluteRCPPath,
	/**
	 * Next argument is relative rcp path.
	 */
	RelativeRCPPath,
	/**
	 * Next parameter is an ods file and should be processed.
	 */
	ReadODS,
	/**
	 * Next parameter is an xml file and should be processed.
	 */
	ReadXML,

    }

    /**
     * Print help text for command line interface
     */
    private static void printHelp() {

	System.out.println("Parameter:-h:       Print this help text.");
	System.out
		.println("          -as <path> Specify a absolute target base path. This path is extended by package structure. Default is ods-file's path + 'source/main/java'.");
	System.out
		.println("          -rs <path> Specify a relative target base path. This path is relative to user.dir");
	System.out
		.println("          -ae <path> Specify a absolute rcp plugin base path. Default is path of ods file.");
	System.out
		.println("          -re <path> Specify a relative rcp plugin base path. This path is relative to user.dir.");

	System.out
		.println("          -axe <file> Export loaded tables in xml format. Specify target absolute.");
	System.out
		.println("          -rxe <file> Export loaded tables in xml format. Specify target absolute.");

	System.out
		.println("          -jdk <4/5>       Create source code for jdk 1.4 or 1.5. Default is 1.5.");
	System.out
		.println("          -server          Create source for server environment, default is client.");
	System.out
		.println("          -client          Create source for client environment, this is the default .");
	System.out
		.println("          -rcp <RCP Root>  Set the Eclipse-RCP base path needed to write i18n property files.");
	System.out
		.println("          -ods <file>      Specify OpenOffice Calc (ODS) file for processing.");
	System.out
		.println("          -xml <file>      Specify XML file for processing.");
	System.out
		.println("          -codegen      Generate code with parameters specified before.");
	System.out
		.println("Parameters can be added in any order. A target path is used until the next is specified.");
    }

}
