/**
 * Copyright 2005-2008 Olaf Panz 
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 *
 * 		http://www.apache.org/licenses/LICENSE-2.0 
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 */
package de.olafpanz.translationtable;

import java.io.File;
import java.util.EnumSet;
import java.util.Set;

/**
 * Handles parameters for translation table code generator. Handles also default
 * values.
 * 
 * @author Olaf Panz
 * 
 */
public class TranslationTableParameters {

    /**
     * Flags that describes currently set default values
     * 
     * @author Olaf Panz
     * 
     */
    private enum DefaultFlags {
	/**
	 * Marks that the source root path has default value.
	 */
	SourceRootIsDefault,
	/**
	 * Marks that rcp plugin path has default.
	 */
	RcpPluginIsDefault,
    }

    /**
     * Marks values that have currently the default value.
     */
    private final Set<DefaultFlags> defaults = EnumSet
	    .noneOf(DefaultFlags.class);

    /**
     * Root of source code that should be generated.
     */
    private File sourceRoot;
    /**
     * Root of rcp plugin, used to write eclipse-rcp translations
     */
    private File rcpPluginRoot;
    /**
     * Name of ods or xml source-file to process.
     */
    private File file;

    /**
     * Check, if all parameters are set
     * 
     * @return Error message on error.
     */
    public String isComplete() {
	if (file == null) {
	    return "No file is set for processing.";
	}
	if (!file.exists()) {
	    return "File '" + file + "' does not exist.";
	}
	return null;
    }

    /**
     * JDK to generate for.
     */
    private JDKDefine jdk;
    /**
     * Environment to generate for.
     */
    private Environment env;

    /**
     * Create class with defaults.
     */
    public TranslationTableParameters() {
	setDefault();
    }

    /**
     * Init with all parameters.
     * 
     * @param env
     *                Environment to generate for.
     * @param jdk
     *                JDK to generate for.
     * @param ods
     *                Name of ods file to process.
     * @param rcpPluginRoot
     *                Root of rcp plugin, used to write eclipse-rcp translations
     * @param sourceRoot
     *                Root of source code that should be generated.
     */
    public TranslationTableParameters(final Environment env,
	    final JDKDefine jdk, final File ods, final File rcpPluginRoot,
	    final File sourceRoot) {
	this();
	setOds(ods);
	setEnvironment(env);
	setJdk(jdk);
	setRcpPluginRoot(rcpPluginRoot);
	setSourceRoot(sourceRoot);
    }

    /**
     * Set default values for all parameters
     */
    public void setDefault() {
	sourceRoot = new File("user.dir");
	rcpPluginRoot = sourceRoot;
	file = null;
	jdk = JDKDefine.JDK15;
	env = Environment.Client;

	defaults.addAll(EnumSet.allOf(DefaultFlags.class));
    }

    /**
     * @return Root of source code that should be generated.
     */
    public File getSourceRoot() {
	return sourceRoot;
    }

    /**
     * @param sourceRoot
     *                Root of source code that should be generated.
     */
    public void setSourceRoot(final File sourceRoot) {
	if (sourceRoot == null) {
	    throw new IllegalArgumentException("Set null to source root");
	}
	defaults.remove(DefaultFlags.SourceRootIsDefault);
	this.sourceRoot = sourceRoot;
    }

    /**
     * @return Root of rcp plugin, used to write eclipse-rcp translations
     */
    public File getRcpPluginRoot() {
	return rcpPluginRoot;
    }

    /**
     * @param rcpPluginRoot
     *                Root of rcp plugin, used to write eclipse-rcp translations
     */
    public void setRcpPluginRoot(final File rcpPluginRoot) {
	if (rcpPluginRoot == null) {
	    throw new IllegalArgumentException("Set null to rcp-plugin-root");
	}
	defaults.remove(DefaultFlags.RcpPluginIsDefault);
	this.rcpPluginRoot = rcpPluginRoot;

    }

    /**
     * Set the default rcp plugin path, call will not change defaults-flags
     * 
     * @param rcpPluginRoot
     *                Null is not allowed.
     */
    private void setDefaultRcpPluginRoot(final File rcpPluginRoot) {
	if (defaults.contains(DefaultFlags.RcpPluginIsDefault)) {
	    this.rcpPluginRoot = rcpPluginRoot;
	}
    }

    /**
     * Set default source root from given plugin root. Is only set, if
     * source-root has still a default value
     * 
     * @param rcpPluginRoot
     *                Null is not allowed.
     */
    private void setDefaultSourceRoot(final File rcpPluginRoot) {

	if (defaults.contains(DefaultFlags.SourceRootIsDefault)) {

	    final File defSourceRoot = new File(rcpPluginRoot, "source"
		    + File.separator + "main" + File.separator + "java");
	    sourceRoot = defSourceRoot;
	}
    }

    /**
     * @return Name of ods file to process.
     */
    public File getFile() {
	return file;
    }

    /**
     * @param file
     *                Name of ods or xml file to process.
     */
    public void setXml(final File file) {
	if (file == null) {
	    throw new IllegalArgumentException("Set null to xml-file.");
	}
	if (!file.exists()) {
	    throw new IllegalArgumentException("Specified XML file '" + file
		    + "' does not exists.");
	}

	setDefaultRcpPluginRoot(file.getParentFile());
	setDefaultSourceRoot(file.getParentFile());

	this.file = file;
    }

    /**
     * @param file
     *                Name of ods or xml file to process.
     */
    public void setOds(final File file) {
	if (file == null) {
	    throw new IllegalArgumentException("Set null to ods-file.");
	}
	if (!file.exists()) {
	    throw new IllegalArgumentException("Specified ODS file '" + file
		    + "' does not exists.");
	}

	setDefaultRcpPluginRoot(file.getParentFile());
	setDefaultSourceRoot(file.getParentFile());

	this.file = file;
    }

    /**
     * @return JDK to generate for.
     */
    public JDKDefine getJdk() {
	return jdk;
    }

    /**
     * @param jdk
     *                JDK to generate for.
     */
    public void setJdk(final JDKDefine jdk) {
	if (jdk == null) {
	    throw new IllegalArgumentException("jdk is null");
	}

	this.jdk = jdk;
    }

    /**
     * @return Environment to generate for.
     */
    public Environment getEnvironment() {
	return env;
    }

    /**
     * @param env
     *                Environment to generate for.
     */
    public void setEnvironment(final Environment env) {
	if (env == null) {
	    throw new IllegalArgumentException("Env is null.");
	}

	this.env = env;
    }
}
